/*
 * sngtc_cfg
 * Sangoma Transcoder Configuration Utility
 *
 * Nenad Corbic <ncorbic@sangoma.com>
 * Moises Silva <moy@sangoma.com>
 *
 * Copyright (C) 2010 Sangoma Technologies
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * * Neither the name of the original author; nor the names of any contributors
 * may be used to endorse or promote products derived from this software
 * without specific prior written permission.
 * 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Contributors:
 *
 */

#define _GNU_SOURCE
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <time.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <arpa/inet.h>
#include <errno.h>
#include <getopt.h>
#include <sng_tc/sng_tc.h>

enum {
	SNGTC_LOGLEVEL_L_INFO = SNGTC_LOGLEVEL_PUBLIC,
	SNGTC_LOGLEVEL_L_RES
};

#define ARRLEN(arr) (sizeof(arr)/sizeof(arr[0]))

#define PROD "sngtc_cfg"

#undef SNGTC_PRINT
#undef SNGTC_DEBUG
#undef SNGTC_WARN
#undef SNGTC_ERROR
#undef SNGTC_INFO
#undef SNGTC_CRIT

#define SNGTC_PRINT(level,a,...) log_printf(level,a,##__VA_ARGS__)
#define SNGTC_DEBUG(a,...) 		SNGTC_PRINT(SNGTC_LOGLEVEL_DEBUG, PROD ": " a, ##__VA_ARGS__)
#define SNGTC_WARN(a,...) 		SNGTC_PRINT(SNGTC_LOGLEVEL_WARN, PROD ": " a, ##__VA_ARGS__)

#define SNGTC_ERROR(a,...)		SNGTC_PRINT(SNGTC_LOGLEVEL_ERROR, PROD ": " a, ##__VA_ARGS__)
#define SNGTC_CRIT(a,...) 		SNGTC_PRINT(SNGTC_LOGLEVEL_CRIT,a, ##__VA_ARGS__)

#define SNGTC_RES(a,...) 		SNGTC_PRINT(SNGTC_LOGLEVEL_L_RES,a, ##__VA_ARGS__)

#undef SNGTC_INFO
#define SNGTC_INFO(a,...) 		SNGTC_PRINT(SNGTC_LOGLEVEL_L_INFO,a,##__VA_ARGS__)

#define SNGTC_L_LOGLEVEL_DECODE(level) \
		((level==SNGTC_LOGLEVEL_DEBUG) ? "SNGTC_DEBUG" : \
		(level==SNGTC_LOGLEVEL_WARN) ? "SNGTC_WARN" : \
		(level==SNGTC_LOGLEVEL_INFO) ? "SNGTC_INFO" : \
		(level==SNGTC_LOGLEVEL_L_INFO) ? "SNGTC_INFO" : \
		(level==SNGTC_LOGLEVEL_L_RES) ? "SNGTC_RES" : \
		(level==SNGTC_LOGLEVEL_ERROR) ? "SNGTC_ERROR" : \
		(level==SNGTC_LOGLEVEL_CRIT) ? "SNGTC_CRIT" : "Unkown")

#define SNGTC_SRV_CFG  "sngtc_server.conf.xml"
#define SNGTC_SRV_DEFAULT_DIR	"/etc/sngtc"

#define SNGTC_ASTERISK_CFG "sangoma_codec.conf"
#define SNGTC_ASTERISK_DEFAULT_DIR "/etc/asterisk"

#define SNGTC_FREESWITCH_CFG "sangoma_codec.conf.xml"
#define SNGTC_FREESWITCH_DEFAULT_DIR "/usr/local/freeswitch/conf/autoload_configs"


#define TMP_SUFFIX "tmp"
static struct {
	int do_server;
	int do_asterisk;
	int do_freeswitch;

	char srvdir[512];
	char astdir[512];
	char fsdir[512];

	FILE *srvfile;
	FILE *astfile;
	FILE *fsfile;

	char srvfile_name[1024];
	char astfile_name[1024];
	char fsfile_name[1024];

	int do_color;
	int verbose;
} sngtc_globals;

/* these defines must match the index in the sngtc_long_options array  */
#define OPT_SERVER_IDX 0
#define OPT_CFG_DIR_IDX 1
#define OPT_ASTERISK_IDX 2
#define OPT_FREESWITCH_IDX 3
#define OPT_ASTDIR_IDX 4
#define OPT_FSDIR_IDX 5
#define OPT_VERBOSE_IDX 6
#define OPT_COLOR_IDX 7
static struct option sngtc_long_options[] =
{
	{ "server", 0, &sngtc_globals.do_server, 1},
	{ "srvdir", 1, NULL, 0 },
	{ "asterisk", 0, &sngtc_globals.do_asterisk, 1},
	{ "freeswitch", 0, &sngtc_globals.do_freeswitch, 1},
	{ "astdir", 1, NULL, 0 },
	{ "fsdir", 1, NULL, 0 },
	{ "verbose", 1, &sngtc_globals.verbose,  1},
	{ "color", 0, &sngtc_globals.do_color , 1},
	{ 0, 0, 0, 0 },
};
#define SNGTC_OPT_STRING ""

#define MAX_ETH_DEVS 32
system_eth_cfg_t sys_cfg[MAX_ETH_DEVS];
uint32_t sys_cfg_sz = 0;

#define SNG_RED "\033[22;31m"
#define SNG_BLACK "\033[22;30m"
#define SNG_GREEN "\033[22;32m"
enum {
	SNG_COLOR_BLACK,
	SNG_COLOR_RED,
	SNG_COLOR_GREEN
};

static int log_printf(int level, char *fmt, ...)
{
	char *data;
	int ret = 0;
	va_list ap;
	char date[200] = "";

	struct tm now;
	time_t epoch;

	switch (level) {
	case SNGTC_LOGLEVEL_DEBUG:
		if (sngtc_globals.verbose < 10) {
			return -1;
		}
	case SNGTC_LOGLEVEL_WARN:
	case SNGTC_LOGLEVEL_INFO:
		if (sngtc_globals.verbose == 0) {
			return -1;
		}
	case SNGTC_LOGLEVEL_L_INFO:
	case SNGTC_LOGLEVEL_L_RES:
	case SNGTC_LOGLEVEL_ERROR:
	case SNGTC_LOGLEVEL_CRIT:
		break;
	default:
		fprintf(stderr, "%s: invalid loglevel %i\n",__FUNCTION__,level);
		return -1;
	}

	if (time(&epoch) && localtime_r(&epoch, &now)) {
		strftime(date, sizeof(date), "%Y-%m-%d %T", &now);
	}

	va_start(ap, fmt);
	ret = vasprintf(&data, fmt, ap);
	if (ret == -1) {
		return ret;
	}
        
	if (!sngtc_globals.do_color) {
		printf("[%-11s] %s", SNGTC_L_LOGLEVEL_DECODE(level), data);
	} else {
		switch (level) {
		case SNGTC_LOGLEVEL_ERROR:
		case SNGTC_LOGLEVEL_CRIT:
			printf(SNG_RED "[%-11s] %s", SNGTC_L_LOGLEVEL_DECODE(level), data);
			break;
		case SNGTC_LOGLEVEL_L_RES:
			printf(SNG_GREEN "[%-11s] %s", SNGTC_L_LOGLEVEL_DECODE(level), data);
			break;
		default:
			printf(SNG_BLACK "[%-11s] %s", SNGTC_L_LOGLEVEL_DECODE(level), data);
			break;
		}
	}

	free(data);

	return 0;
}

static int sngtc_write_sngtc_server_file_begin()
{
	FILE *f = sngtc_globals.srvfile;
	fprintf(f,"<configuration name=\"sngtc_server.conf\" description=\"Sangoma Transcoding Server Configuration\">\n\n");

	fprintf(f,"\t<vocallos>\n\n");
	return 0;
}

static int sngtc_write_freeswich_file_begin(void)
{
	FILE *f = sngtc_globals.fsfile;

	fprintf(f,"<configuration name=\"sangoma_codec.conf\" description=\"Sangoma Codec Configuration\">\n\n");

	fprintf(f,"\t<settings>\n" 
			"\t\t<!-- Please check http://wiki.sangoma.com/sangoma-media-transcoding for documentation on the supported options -->\n");

	return 0;
}

static int sngtc_write_sngtc_server_file_end(void)
{
	FILE *f = sngtc_globals.srvfile;

	fprintf(f, "\t</vocallos>\n\n");

	fprintf(f, "</configuration>\n\n");

	return 0;
}
static int sngtc_write_freeswich_file_end(void)
{
	FILE *f = sngtc_globals.fsfile;

	fprintf(f, "\t</settings>\n\n");

	fprintf(f, "</configuration>\n\n");

	return 0;
}

static int sntc_write_sngtc_server_file(system_eth_cfg_t *e_cfg, sngtc_nic_vocallo_map_t *nic_map)
{
	FILE *f = sngtc_globals.srvfile;
	uint32_t base_ip = nic_map->voc_base_ip.ip;
       	uint8_t octet = (0xFF & base_ip);
	fprintf(f, "\t\t<vocallo name=\"%s\">\n\n"
			"\t\t\t<!-- Starting UDP port for the vocallo -->\n"
			"\t\t\t<param name=\"base_udp\" value=\"%d\"/>\n"
			"\t\t\t<!-- Starting final octet for the IP addresses to use for the vocallo modules  -->\n"
			"\t\t\t<param name=\"base_ip_octet\" value=\"%d\"/>\n\n"
		    "\t\t</vocallo>\n\n",  e_cfg->name, nic_map->voc_base_udp_prot, octet);
	return 0;
}

static int sngtc_write_asterisk_file_begin(void)
{
	FILE *f = sngtc_globals.astfile;
	fprintf(f, "[general]\n"
		"; Please check http://wiki.sangoma.com/sangoma-media-transcoding for documentation on other supported options\n"
		"; Attempt to avoid codec conflicts in Asterisk that can cause Asterisk\n"
		"; to crash on codec module load\n");
	fprintf(f, "unregisterconflictingcodecs=yes\n\n");

	fprintf(f, "; by default we only register G729. Type 'sangoma show codecs' at the Asterisk CLI for a list of supported codecs.\n");
	fprintf(f, "; you can also register specific translation paths, like g729toulaw,ulawtog729. If only the codec is specified all possible\n");
	fprintf(f, "; translation combinations for that codec will be registered\n");
	fprintf(f, "; type 'sangoma show translators' at the Asterisk CLI for the list of translators registered\n");
	fprintf(f, "register=g729\n\n");
	return 0;
}

static int sngtc_write_asterisk_rtp_ip(const char *ip)
{
	FILE *f = sngtc_globals.astfile;
	fprintf(f, "; This IP address will be used for RTP. It must be reachable by the vocallo network.\n");
	fprintf(f, "rtpip=%s\n\n", ip);
	return 0;
}

#define TMP_FILE_FORMAT "%s.tmp"
static int sngtc_create_cfgs(void)
{
	FILE *f = NULL;
	char tmpfilename[sizeof(sngtc_globals.srvfile_name)+10];

	if (sngtc_globals.do_server) {
		snprintf(tmpfilename, sizeof(tmpfilename), TMP_FILE_FORMAT, sngtc_globals.srvfile_name);
		f = fopen(tmpfilename, "w");
		if (!f) {
			SNGTC_CRIT("Failed to create server file %s!\n", tmpfilename);
			return -1;
		}
		sngtc_globals.srvfile = f;
		sngtc_write_sngtc_server_file_begin();
	}

	if (sngtc_globals.do_asterisk) {
		snprintf(tmpfilename, sizeof(tmpfilename), TMP_FILE_FORMAT, sngtc_globals.astfile_name);
		f = fopen(tmpfilename, "w");
		if (!f) {
			SNGTC_CRIT("Failed to create Asterisk transcoding configuration file %s!\n", tmpfilename);
			return -1;
		}
		sngtc_globals.astfile = f;
		sngtc_write_asterisk_file_begin();
	}

	if (sngtc_globals.do_freeswitch) {
		snprintf(tmpfilename, sizeof(tmpfilename), TMP_FILE_FORMAT, sngtc_globals.fsfile_name);
		f = fopen(tmpfilename, "w");
		if (!f) {
			SNGTC_CRIT("Failed to create FreeSWITCH transcoding configuration file %s!\n", tmpfilename);
			return -1;
		}
		sngtc_globals.fsfile = f;
		sngtc_write_freeswich_file_begin();
	}
	return 0;
}

static void print_usage(void)
{
	SNGTC_INFO("Usage: sngtc_cfg supports the following options:\n\n");
	SNGTC_INFO("--server - Creates a configuration for the Sangoma transcoding daemon. Not needed if you are only configuring the client side.\n");
	SNGTC_INFO("--asterisk - Creates a configuration for Asterisk.\n");
	SNGTC_INFO("--freeswitch - Creates a configuration for FreeSWITCH.\n");
	SNGTC_INFO("--srvdir=<path> - Where to place the server configuration.\n");
	SNGTC_INFO("--astdir=<path> - Where to place the Asterisk configuration.\n");
	SNGTC_INFO("--fsdir=<path> - Where to place the FreeSWITCH configuration.\n");
	SNGTC_INFO("--verbose [value between 0 and 100] - Verbosity of the logging printed.\n");
	SNGTC_INFO("--color - Enable colored output.\n");
	SNGTC_INFO("\n");
}

/*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*\

Function:       main

Description:    Main program.

\*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*/
int main(int argc, char *argv[])
{
	int i;
	int x;
	int err;
	char cmdline[2048];
	char res_str[1024];
	int res_int = 0;
	int  ip_octet, orig_octet;
	sngtc_init_cfg_t init_cfg;
	system_eth_cfg_t *eth_cfg;
	char in[1024];
	int dev_cfg = 0;
	int detect = 0;
	int detect_cnt = 0;
	int optindex = 0;
	int optret = 0;
	int rtpip_set = 0;
	
	sngtc_nic_vocallo_map_t *nic_voc_map;
	sngtc_vocallo_map_t *voc_map;
	uint32_t nic_voc_map_sz;

	if (argc < 2) {
		print_usage();
		exit(-1);
	}
	
	/* Set some default values in case the options are not present for them */
	memset(&sngtc_globals, 0, sizeof(sngtc_globals));

	snprintf(sngtc_globals.srvdir, sizeof(sngtc_globals.srvdir), SNGTC_SRV_DEFAULT_DIR);
	snprintf(sngtc_globals.srvfile_name, sizeof(sngtc_globals.srvfile_name), SNGTC_SRV_DEFAULT_DIR "/%s", SNGTC_SRV_CFG);

	snprintf(sngtc_globals.astdir, sizeof(sngtc_globals.astdir), SNGTC_ASTERISK_DEFAULT_DIR);
	snprintf(sngtc_globals.astfile_name, sizeof(sngtc_globals.astfile_name), SNGTC_ASTERISK_DEFAULT_DIR "/%s", SNGTC_ASTERISK_CFG);

	snprintf(sngtc_globals.fsdir, sizeof(sngtc_globals.fsdir), SNGTC_FREESWITCH_DEFAULT_DIR);
	snprintf(sngtc_globals.fsfile_name, sizeof(sngtc_globals.fsfile_name), SNGTC_FREESWITCH_DEFAULT_DIR "/%s", SNGTC_FREESWITCH_CFG);

	system("clear");

	SNGTC_INFO("\n");
	SNGTC_INFO("====================================================================\n");

	/* we don't want opt errors printed to stderr */
	opterr = 0;
	for ( ; ; ) {

		optindex = -1;
		optret = getopt_long_only(argc, argv, SNGTC_OPT_STRING, sngtc_long_options, &optindex);

		if (optret == -1) {
			/* end if CLI args */
			break;
		}

		if (optret == '?') {
			SNGTC_ERROR("Unrecognized command line option\n");
			exit(-1);
		}

		/*SNGTC_ERROR("got option %s\n", sngtc_long_options[optindex].name);*/
		switch (optindex) {
		case OPT_CFG_DIR_IDX:
			snprintf(sngtc_globals.srvdir, sizeof(sngtc_globals.srvdir), "%s", optarg);
			snprintf(sngtc_globals.srvfile_name, sizeof(sngtc_globals.srvfile_name), "%s/%s", optarg, SNGTC_SRV_CFG);
			break;

		case OPT_ASTDIR_IDX:
			snprintf(sngtc_globals.astdir, sizeof(sngtc_globals.astdir), "%s", optarg);
			snprintf(sngtc_globals.astfile_name, sizeof(sngtc_globals.astfile_name), "%s/%s", optarg, SNGTC_ASTERISK_CFG);
			break;

		case OPT_FSDIR_IDX:
			snprintf(sngtc_globals.fsdir, sizeof(sngtc_globals.fsdir), "%s", optarg);
			snprintf(sngtc_globals.fsfile_name, sizeof(sngtc_globals.fsfile_name), "%s/%s", optarg, SNGTC_FREESWITCH_CFG);
			break;

		case OPT_SERVER_IDX:
		case OPT_ASTERISK_IDX:
		case OPT_FREESWITCH_IDX:
		case OPT_VERBOSE_IDX:
		case OPT_COLOR_IDX:
			/* nothing to do for this options, getopt already set the integer option */
			break;

		default:
			SNGTC_CRIT("Unhandled option %s/%d\n", sngtc_long_options[optindex].name, optindex);
			exit(-1);
			break;
		}
	}

	if (sngtc_globals.do_server) {
		SNGTC_INFO("Configuring Sangoma Transcoding Server\n");
	}	

	if (sngtc_globals.do_asterisk) {
		SNGTC_INFO("Configuring Sangoma Transcoding for Asterisk\n");
	}

	if (sngtc_globals.do_freeswitch) {
		SNGTC_INFO("Configuring Sangoma Transcoding for FreeSWITCH\n");
	}
	
	SNGTC_INFO("\n");

	err = sngtc_create_cfgs();
	if (err) {
		SNGTC_ERROR("Failed to create requested configuration\n");
		return err;
	}

	if (!sngtc_globals.do_server) {
		SNGTC_INFO("Skipping detection since --server option was not specified\n");
		goto done;
	}

	
	sleep(1);

	err = sngtc_detect_eth_dev();
	if (!err) {
		SNGTC_ERROR("Failure detecting eth devices\n");
		return -1;
	}


	SNGTC_DEBUG("Detected %i eth devices\n", err);
	memset(&init_cfg, 0, sizeof(init_cfg));
	init_cfg.log = log_printf;

	SNGTC_INFO("\n");
	SNGTC_INFO("Detecting Transcoding Card & Modules ...\n", nic_voc_map_sz);

	err = sngtc_get_nic_vocallo_map(&init_cfg, &nic_voc_map, &nic_voc_map_sz);
	if (err) {
		SNGTC_ERROR("Failed to detect transcoding modules!\n");
		SNGTC_INFO("\n");
		return err;
	}

retry_again:

	detect++;

	SNGTC_INFO("\n");
	SNGTC_INFO("Detected Transcoding Card & Modules: %d\n", nic_voc_map_sz);
	SNGTC_INFO("\n");

	detect_cnt = 0;

	for (i = 0; i < (int)nic_voc_map_sz; i++) {

		detect_cnt++;
		if (nic_voc_map[i].init > 1) {
			continue;
		}

		err = sngtc_find_ethconf(NULL, nic_voc_map[i].nic_mac.mac, &eth_cfg);
		if (err) {
			SNGTC_ERROR("Failed to retrieve configuration of device with mac %s\n", nic_voc_map[i].nic_mac.mac_str);
			continue;
		}

		SNGTC_INFO("[%d]: %s: Mac=%s IP=%s/%s\n", i + 1, eth_cfg->name, nic_voc_map[i].nic_mac.mac_str, eth_cfg->ip.ip_str, eth_cfg->mask.ip_str);

		SNGTC_INFO("--------------------------------------------------------\n");
		if (nic_voc_map[i].voc_mods_sz) {
			for (x =0 ; x < nic_voc_map[i].voc_mods_sz; x++) {
				voc_map = &nic_voc_map[i].voc_map[x];
				SNGTC_INFO("\t-> Transcoding-Module %i: Mac=%s Error=%i\n", x + 1, voc_map->v_mac.mac_str, voc_map->error);
			}
			if (i != (int)nic_voc_map_sz) {
				SNGTC_INFO("\n");
			}
		}
	}
         
	SNGTC_INFO("--------------------------------------------------------\n");
	SNGTC_INFO("\n");
	SNGTC_INFO("Please select device to configure: [1-%i] q:quit ", detect_cnt);

	fgets(in, sizeof(in), stdin);
	sscanf(in, "%s", res_str);
	if (in[0] == 'q') {
		goto done;
	}

	i = atoi(in);
	if (i > 0 && i <= detect_cnt) {

		i--;
		
		err = sngtc_find_ethconf(NULL, nic_voc_map[i].nic_mac.mac, &eth_cfg);
		if (err || !eth_cfg) {
			SNGTC_CRIT("%s: Failed to detect eth device with mac %s - should never happen\n", PROD,nic_voc_map[i].nic_mac.mac_str);
			/* we cannot recover here because this should never fail */
			exit(1);
		}

		if (nic_voc_map[i].init > 1) {
			SNGTC_ERROR("Device %s already configured!\n", eth_cfg->name);
			goto retry_again;
		}
		nic_voc_map[i].init++;
		dev_cfg++;

		SNGTC_INFO("--------------------------------------------------------\n");
		SNGTC_INFO("Configuring %s: Mac=%s IP=%s/%s %p\n",
			 eth_cfg->name,
			 nic_voc_map[i].nic_mac.mac_str,
			 eth_cfg->ip.ip_str,
			 eth_cfg->mask.ip_str,
			 &nic_voc_map[i]);


		nic_voc_map[i].voc_base_ip.ip = eth_cfg->ip.ip;
		nic_voc_map[i].voc_base_ip.ip &= 0xFFFFFF00;
		nic_voc_map[i].voc_base_ip.ip |= 0x00000064+50;
		sngtc_codec_ipv4_hex_to_str(nic_voc_map[i].voc_base_ip.ip,nic_voc_map[i].voc_base_ip.ip_str);

		nic_voc_map[i].voc_base_netmask = eth_cfg->mask;

retry:
		SNGTC_INFO("\n");
		SNGTC_INFO("\t----------------------------------------------------------\n");
		SNGTC_INFO("\tPlease specify the base IP range for Transcoding Modules:\n");
		SNGTC_INFO("\tDefault IP value: %s/%s \n", nic_voc_map[i].voc_base_ip.ip_str, nic_voc_map[i].voc_base_netmask.ip_str);
		SNGTC_INFO("\n");

		SNGTC_INFO("\tSpecify last octet of the IP addres: [1-255] or <enter> for default:\n");
		SNGTC_INFO("\t#> ");
		memset(res_str, 0, sizeof(res_str));

		fgets(in, sizeof(in), stdin);
		sscanf(in, "%s", res_str);
		if ((ip_octet = atoi(res_str)) > 0) {
			orig_octet = eth_cfg->ip.ip & 0xFF;
			err = 0;
			
			SNGTC_DEBUG("\tIP orig %i new %i\n", orig_octet, ip_octet);
			if (orig_octet == ip_octet) {
				err++;	
			} 
			
			if (orig_octet > ip_octet) {
				if (!(orig_octet - nic_voc_map[i].voc_mods_sz > 0 &&
				      orig_octet - nic_voc_map[i].voc_mods_sz > ip_octet)) {
					err++;
				}
			} 
			
			if (orig_octet < ip_octet) {
				if (!(ip_octet > orig_octet && (255 - (ip_octet > nic_voc_map[i].voc_mods_sz)))) {
					err++;
				}	 
			}
						
			if (err == 0) {	 
				
				nic_voc_map[i].voc_base_ip.ip &= 0xFFFFFF00;
				nic_voc_map[i].voc_base_ip.ip |= ip_octet;
				sngtc_codec_ipv4_hex_to_str(nic_voc_map[i].voc_base_ip.ip, nic_voc_map[i].voc_base_ip.ip_str);
			
			} else {
				SNGTC_ERROR("\tSelected Transcoding Module IP/Mask = %s/%s\n", eth_cfg->ip.ip_str, nic_voc_map[i].voc_base_netmask.ip_str);
				SNGTC_ERROR("\tTranscoding Module Base IP cannot be the same as or within range of %s IP %s\n", eth_cfg->name,eth_cfg->ip.ip_str);
				goto retry;
			}
		} 

		nic_voc_map[i].voc_base_udp_prot = 15000;

		SNGTC_RES("\tSelected Transcoding Module IP/Mask = %s/%s \n", nic_voc_map[i].voc_base_ip.ip_str, nic_voc_map[i].voc_base_netmask.ip_str);

		SNGTC_INFO("\n");
		SNGTC_INFO("\t----------------------------------------------------------\n");
		SNGTC_INFO("\tPlease specify the base UDP port for Transcoding Modules:\n");
		SNGTC_INFO("\tDefault udp base value  : %d \n", nic_voc_map[i].voc_base_udp_prot);
		SNGTC_INFO("\n");
		SNGTC_INFO("\tSpecify udp base value [9000-65000] or <enter> for default:\n");
		SNGTC_INFO("\t#> ");
		
		res_int = 0;
		fgets(in, sizeof(in), stdin);
		sscanf(in, "%d", &res_int);

		if (res_int >= 9000) {
			nic_voc_map[i].voc_base_udp_prot=res_int;
		}
		SNGTC_RES("\tSelected UDP Port Base %d\n", nic_voc_map[i].voc_base_udp_prot);
		SNGTC_INFO("\n");
		SNGTC_INFO("====================================================================\n");
		SNGTC_INFO("\n");

		if (sngtc_globals.srvfile) {
			err = sntc_write_sngtc_server_file(eth_cfg, &nic_voc_map[i]);	
			if (err) {
				SNGTC_INFO("Failed to write server file\n");
				exit(-1);
			}
		}

		if (sngtc_globals.astfile && !rtpip_set) {
			err = sngtc_write_asterisk_rtp_ip(eth_cfg->ip.ip_str);	
			if (err) {
				SNGTC_INFO("Failed to write asterisk file\n");
				exit(-1);
			}
			rtpip_set++;
		}

		if (dev_cfg != detect_cnt) {
			goto retry_again;
		}
	} else {
		SNGTC_ERROR("Invalid option selected\n");
     		goto retry_again;
	}

done:
	if (sngtc_globals.srvfile) {
		sngtc_write_sngtc_server_file_end();
		fclose(sngtc_globals.srvfile);
		snprintf(cmdline, sizeof(cmdline), "mv " TMP_FILE_FORMAT " %s", sngtc_globals.srvfile_name, sngtc_globals.srvfile_name);
		system(cmdline);
		SNGTC_INFO("\n");
		SNGTC_INFO("Sangoma Transcoding configuration file saved: %s\n", sngtc_globals.srvfile_name);
		SNGTC_INFO("\n");
	} 

	if (sngtc_globals.astfile) {
		fclose(sngtc_globals.astfile);
		snprintf(cmdline, sizeof(cmdline), "mv " TMP_FILE_FORMAT " %s", sngtc_globals.astfile_name, sngtc_globals.astfile_name);
		system(cmdline);
		SNGTC_INFO("\n");
		SNGTC_INFO("Asterisk Sangoma Transcoding configuration file saved: %s\n", sngtc_globals.astfile_name);
		SNGTC_INFO("\n");
	}

	if (sngtc_globals.fsfile) {
		sngtc_write_freeswich_file_end();
		fclose(sngtc_globals.fsfile);
		snprintf(cmdline, sizeof(cmdline), "mv " TMP_FILE_FORMAT " %s", sngtc_globals.fsfile_name, sngtc_globals.fsfile_name);
		system(cmdline);
		SNGTC_INFO("\n");
		SNGTC_INFO("FreeSWITCH Sangoma Transcoding configuration file saved: %s\n", sngtc_globals.fsfile_name);
		SNGTC_INFO("\n");
	}

    return 0;
}

