/*
 * Sangoma Transcoder SOAP Server
 *
 * Copyright (C) 2010, Sangoma Technologies 
 *
 * Nenad Corbic <ncorbic@sangoma.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *
 * Moises Silva <moy@sangoma.com>
 *
 */

#include "sngtc_server.h"

#include <libxml/parser.h>
#include <libxml/tree.h>
#include <libxml/xmlmemory.h>

#ifdef LIBXML_TREE_ENABLED

static int host_nic_vocallo_idx = -1;

static int parse_settings(xmlNode *cur)
{
	xmlChar *name, *value;
	for (cur = cur->children; cur; cur = cur->next) {
		if (cur->type != XML_ELEMENT_NODE) {
			continue;
		}
		if (xmlStrcmp(cur->name, (const xmlChar *) "param")) {
			continue;
		}

		name = xmlGetProp(cur, (const xmlChar *) "name");
		value = xmlGetProp(cur, (const xmlChar *) "value");

		if (!name || !value) {
			SNGTC_ERROR("XML parameter missing either name or value\n");
			return -1;
		}

		if (!xmlStrcmp(name, (const xmlChar *) "bindaddr")) {
			strncpy(sngtc_server_ip_str, (char *)value, sizeof(sngtc_server_ip_str));
			sngtc_server_ip_str[sizeof(sngtc_server_ip_str)-1] = '\0';
			SNGTC_INFO("Found SOAP bindaddr = %s\n", sngtc_server_ip_str);
		}

		if (!xmlStrcmp(name, (const xmlChar *) "bindport")) {
			sngtc_server_port = atoi((char*)value);
			SNGTC_INFO("Found SOAP bindport = %d\n", sngtc_server_port);
		}

		if (!xmlStrcmp(name, (const xmlChar *) "cpuhigh")) {
			sngtc_cpu_high = atoi((char*)value);
			SNGTC_INFO("Found CPU high threshold = %d\n", sngtc_cpu_high);
		}

		if (!xmlStrcmp(name, (const xmlChar *) "cpulow")) {
			sngtc_cpu_low = atoi((char*)value);
			SNGTC_INFO("Found CPU low threshold = %d\n", sngtc_cpu_low);
		}
	}
	return 0;
}

static int parse_vocallo(xmlNode *cur)
{
	int err;
	xmlChar *name, *value;
	system_eth_cfg_t *eth_cfg=NULL;
	uint32_t netmask = 0xFFFFFF00;

	SNGTC_DEBUG(" cur->name = %s\n", cur->name);
	
	host_nic_vocallo_idx++;
	name = xmlGetProp(cur, (const xmlChar *) "name");
	err = sngtc_find_ethconf((char *)name, NULL, &eth_cfg);
	if (err) {
		SNGTC_ERROR("Error: failed to find eth dev %s\n",name);
		return -1;
	}
	SNGTC_INFO("Eth Dev %s mac %s\n",name,eth_cfg->mac.mac_str);

	sprintf((char*)g_init.host_nic_vocallo_cfg[host_nic_vocallo_idx].host_mac.mac_str,eth_cfg->mac.mac_str);

	sngtc_codec_ipv4_str_hex(eth_cfg->ip.ip_str, &g_init.host_nic_vocallo_cfg[host_nic_vocallo_idx].host_ip);
	sngtc_codec_ipv4_str_hex(eth_cfg->mask.ip_str, &g_init.host_nic_vocallo_cfg[host_nic_vocallo_idx].host_ip_netmask);
	sngtc_codec_ipv4_str_hex(eth_cfg->mask.ip_str, &g_init.host_nic_vocallo_cfg[host_nic_vocallo_idx].vocallo_ip_netmask);

	netmask = eth_cfg->mask.ip;

	g_init.host_nic_vocallo_sz++;
	g_init.host_nic_vocallo_cfg[host_nic_vocallo_idx].vocallo_ip				= eth_cfg->ip.ip;
	g_init.host_nic_vocallo_cfg[host_nic_vocallo_idx].vocallo_ip 				&= eth_cfg->mask.ip;
	g_init.host_nic_vocallo_cfg[host_nic_vocallo_idx].vocallo_ip 				|= 0x00000064;
	
	g_init.host_nic_vocallo_cfg[host_nic_vocallo_idx].gateway_ip				= eth_cfg->gateway.ip;
	
	g_init.host_nic_vocallo_cfg[host_nic_vocallo_idx].vocallo_base_udp_port 	= 10000;
	g_init.host_nic_vocallo_cfg[host_nic_vocallo_idx].silence_suppression 		= 0;

	for (cur = cur->children; cur; cur = cur->next) {
		if (cur->type != XML_ELEMENT_NODE) {
			continue;
		}
		if (xmlStrcmp(cur->name, (const xmlChar *) "param")) {
			continue;
		}
		name = xmlGetProp(cur, (const xmlChar *) "name");
		value = xmlGetProp(cur, (const xmlChar *) "value");
		if (!name || !value) {
			SNGTC_ERROR("XML parameter missing either name or value\n");
			return -1;
		}

		if ((!xmlStrcmp(name, (const xmlChar *) "baseudp") || !xmlStrcmp(name, (const xmlChar *)"base_udp"))
			      	&& atoi((char*)value) > 0) {
			SNGTC_INFO("Vocallo [%i]:  base_udp %i\n", host_nic_vocallo_idx, atoi((char*)value));
			g_init.host_nic_vocallo_cfg[host_nic_vocallo_idx].vocallo_base_udp_port = atoi((char*)value);
		}

		if (!xmlStrcmp(name, (const xmlChar *) "vocalloaddr")) {
			SNGTC_INFO("Vocallo [%i]:  vocalloaddr %s\n", host_nic_vocallo_idx, value);
			sngtc_codec_ipv4_str_hex((char*)value, &g_init.host_nic_vocallo_cfg[host_nic_vocallo_idx].vocallo_ip);
		}

		if (!xmlStrcmp(name, (const xmlChar *) "vocallo_base_addr") ||
		    !xmlStrcmp(name, (const xmlChar *) "base_ip_octet")
		    ) {
			SNGTC_INFO("Vocallo [%i]:  vocallo_base_addr/base_ip_octet %s\n", host_nic_vocallo_idx, value);
			g_init.host_nic_vocallo_cfg[host_nic_vocallo_idx].vocallo_ip &= eth_cfg->mask.ip;
			g_init.host_nic_vocallo_cfg[host_nic_vocallo_idx].vocallo_ip |= atoi((char*)value) & 0xFF;
		}

		if (!xmlStrcmp(name, (const xmlChar *) "silence_suppression") && atoi((char*)value) > 0) {
			SNGTC_INFO("Vocallo [%i]:  silence_suppression %i\n", host_nic_vocallo_idx,atoi((char*)value));
			g_init.host_nic_vocallo_cfg[host_nic_vocallo_idx].silence_suppression = atoi((char*)value);
		}
	}

	return 0;
}

/**
 * parse_element_names:
 * @a_node: the initial xml node to consider.
 *
 * Prints the names of the all the xml elements
 * that are siblings or children of a given xml node.
 */
static int parse_server_configuration(xmlNode * a_node)
{
	xmlNode *cur_node = NULL;
	xmlNode *vnode = NULL;
	int err = 0;

	for (cur_node = a_node; cur_node; cur_node = cur_node->next) {
		if (cur_node->type == XML_ELEMENT_NODE) {
			if (!xmlStrcmp(cur_node->name, (const xmlChar *) "settings")) {
				err = parse_settings(cur_node);
				if (err) {
					return err;
				}
			}
			if (!xmlStrcmp(cur_node->name, (const xmlChar *) "vocallos")) {
				for (vnode = cur_node->children; vnode; vnode = vnode->next) {

					if (xmlStrcmp(vnode->name, (const xmlChar *) "vocallo")) {
						continue;
					}

					err = parse_vocallo(vnode);
					if (err) {
						return err;
					}
				}
			}
		}
	}

	return err;
}


/**
 * Simple example to parse a file called "file.xml",
 * walk down the DOM, and print the name of the
 * xml elements nodes.
 */
int sngtc_parse_xml_config_file(char *file)
{
	xmlDoc *doc = NULL;
	xmlNode *root_element = NULL;
	int err;

	err = sngtc_detect_eth_dev();
	if (err < 1) {
		return err;
	}

	/*
	* this initialize the library and check potential ABI mismatches
	* between the version it was compiled for and the actual shared
	* library used.
	*/
	LIBXML_TEST_VERSION

	/*parse the file and get the DOM */
	doc = xmlReadFile(file, NULL, 0);
	if (doc == NULL) {
		SNGTC_ERROR("Error: could not parse file %s\n",file); 
		return -1;
	}

	/*Get the root element node */
	root_element = xmlDocGetRootElement(doc);

	if (parse_server_configuration(root_element->children)) {
		SNGTC_ERROR("Error: could not parse file %s\n",file); 
		return -1;
	}

	/*free the document */
	xmlFreeDoc(doc);

	/*
	*Free the global variables that may
	*have been allocated by the parser.
	*/
	xmlCleanupParser();

    return 0;
}
#else
#error "XML Tree Support not compiled in!\n"
#endif

