/*
 * Sangoma Transcoder SOAP Server
 *
 * Copyright (C) 2010, Sangoma Technologies 
 *
 * Nenad Corbic <ncorbic@sangoma.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *
 * Moises Silva <moy@sangoma.com>
 * David Rokhvarg <davidr@sangoma.com>
 *
 */

#include "sngtc_server.h"
#include "soapH.h"
#include "ns1.nsmap"

#if PLATFORM(Windows) 
# pragma comment(lib, "libsng-tc.lib")
# pragma comment(lib, "libxml2.lib")
# pragma comment(lib, "TS2.lib")
#endif

// -----------------------------------------------------------------------------

// global declarations

// for identification purposes in the log file
static const char sourceID[] = "SNGTC";

#if PLATFORM(Windows) 
/* Path to cfg file directory must be re-initialized at run-time
 * becase System directory is not know at compile-time. */
static char cfg_file[MAX_PATH] = "sngtc_server.conf.xml";
#endif

#if PLATFORM(Linux) 
static char cfg_file[100] = "/etc/sngtc/sngtc_server.conf.xml";
#endif

//TODO: should logLevel be in sngtc_server.conf.xml??
static int logLevel = LOG_LEVEL_FATAL;
static int sngloglevel = SNGTC_LOGLEVEL_STATS;


char sngtc_server_ip_str[SERVER_IP_SIZE] = "127.10.10.1";
int sngtc_server_port = 64055;
int sngtc_cpu_high = 90;
int sngtc_cpu_low = 80;


// command-line options
#if 0
static cchar* options[] =
{
   "-loglevel",
   "-msgsize",
   "-port",
   "-threads1",
   "-threads2",
   0
};
#else
static cchar* options[] =
{
	"-loglevel",
	"-conf",
	"-msgsize",
	"-port",
	"-threads1",
	"-threads2",
	0
};
#endif

sngtc_init_cfg_t g_init;


// threads to be run by the framework
static threadfunc sngtc_cli(void*);
static threadfunc sngtc_soap_server(void*);
static void *monitor_vmod(void *vmod_id);

// helper functions
static uint arg_u(int, char*[], const char*, uint);
static uint arg_str(int argc, char* argv[], const char* argStr, char *str, int len);
static void checkCommandLineOptions(int, char**);
static uint numeric(cchar*);
static void usage(void);
static int threads_running=0;

// -----------------------------------------------------------------------------

// convenience printf-like function for formatting purposes

static void format(char * buf, ushort bufLen, cchar* fmt, va_list ap)
{
	buf[0] = '*';                    // (1 byte)  00
	buf[1] = ' ';                    // (1 byte)  01
	server_formatCurrentTime(buf+2); // (8 bytes) 02 03 04 05 06 07 08 09
	buf[10] = ':';                   // (1 byte)  10
	buf[11] = ' ';
	vsnprintf(buf+12, bufLen-12, fmt, ap);
}


#if 0
LOG_LEVEL_FATAL = 0  ,
LOG_LEVEL_ERROR = 10 ,
LOG_LEVEL_WARN  = 20 ,
LOG_LEVEL_INFO  = 30 ,
LOG_LEVEL_DEBUG = 40 ,
LOG_LEVEL_TRACE = 50
#endif


// -----------------------------------------------------------------------------

// printf-like function that writes on stdout an the log file

int log_printf(int level, char* fmt, ...)
{
	char buf[1024];
	va_list ap;

	if (level < sngloglevel) {
		return 0;
	}

#if 0
	switch (level) {
		
		case SNGTC_LOGLEVEL_STATS:
		case SNGTC_LOGLEVEL_DEBUG:
		case SNGTC_LOGLEVEL_WARN:
		case SNGTC_LOGLEVEL_INFO:
		case SNGTC_LOGLEVEL_ERROR:
		case SNGTC_LOGLEVEL_CRIT:
			break;
		default:
			fprintf(stderr, "%s: Error: invalid loglevel %i\n",__FUNCTION__,level);
			return -1;
	}
#endif

	va_start(ap, fmt);
	format(buf, 1024, fmt, ap);
	va_end(ap);

	buf[0] = '*';                    // (1 byte)  00
	buf[1] = ' ';                    // (1 byte)  01
	server_formatCurrentTime(buf+2); // (8 bytes) 02 03 04 05 06 07 08 09
	buf[10] = ':';                   // (1 byte)  10
	buf[11] = ' ';                   // (1 byte)  11

   // uses "server_printf" (and not plain printf) because of contention issues
   // between threads
	if (threads_running) {
		server_printf("[%-11s] %s", SNGTC_LOGLEVEL_DECODE(level), buf);
		// writes the log file
		buf[strlen(buf)-1]='\0';
		server_logInfo("[%-11s] %s", SNGTC_LOGLEVEL_DECODE(level), buf + 12);
	} else {
		printf("[%-11s] %s", SNGTC_LOGLEVEL_DECODE(level), buf);
	}

	return 0;
}


#define MGD_STACK_SIZE 1024*1024

static int launch_thread(sngtc_thread_t *thread_id, void *obj, void * (*func)(void *))
{
#if WIN32
	if (CreateThread (NULL,
		0,
		(unsigned long (__stdcall *)(void *))func,
		(void *)(obj),
		0,
		thread_id)) {
			/* ok */
			return 0;
		}else{
			/* error */
			return -1;
		}
#else
	pthread_attr_t attr;
	int result = -1;

	
	result = pthread_attr_init(&attr);
        //pthread_attr_setinheritsched(&attr, PTHREAD_EXPLICIT_SCHED);
	//pthread_attr_setschedpolicy(&attr, SCHED_RR);
	pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
	pthread_attr_setstacksize(&attr, MGD_STACK_SIZE);

	result = pthread_create(thread_id, &attr, func, obj);
	if (result) {
		SNGTC_ERROR("Error: Creating Thread! %s\n",
					strerror(errno));
	}
	pthread_attr_destroy(&attr);

	return result;
#endif
}      

static uint16_t vmod[100];

int vocallo_init(void)
{
	int err=-1;
	int detected=0;
	int activated=0;
	int i;
	sngtc_thread_t tid;

    /*
	* Perform initialization tasks required by the application.
	*/
	err=sngtc_detect_init_modules(&g_init, &detected);
	if (err) {
		return err;
	}
	
	for (i=1;i<=detected;i++) {
	
		err=sngtc_module_get_instance(i,&vmod[i]);
		if (err) {
			continue;
		}

		err=sngtc_module_reset(vmod[i]);
		if (err != 0) {
			SNGTC_ERROR("%s: Error: Sangoma Transcoding Module failed to reset\n",
					   sngtc_module_mac(vmod[i]));
			continue;
		}
		
		SNGTC_INFO("%s: Reset Started\n",
				sngtc_module_mac(vmod[i]));
	}

	{
		time_t poll_begin, poll_end;
		unsigned int num_of_seconds_resets_take;
		int reset_done[100];
		int job_finished = 0;

		memset(&reset_done, 0, sizeof(reset_done));

		// Record the start of polling here
		time(&poll_begin);

		// Start polling for all the detected modules until all are available
		// if any module fails to be detected back to normal after 40 seconds, return error
		do {
			int module_reset_finished = 0;
			int j;

			for (j=1; j<=detected; j++) {

				if (reset_done[j]) {
					module_reset_finished++;
				} else {
					// poll for the availability of this module
					if (!sngtc_poll_module(sngtc_module_mac(vmod[j]))) { // successful
						reset_done[j] = 1;
						module_reset_finished++;
						SNGTC_INFO("%s: Reset Finished\n",
								sngtc_module_mac(vmod[j]));
					}
				}
			}

			if (module_reset_finished == detected) {
				job_finished = 1;
			} else {
				usleep(500000);
			}

			time(&poll_end);
			num_of_seconds_resets_take = (unsigned int)difftime(poll_end, poll_begin);

		} while(!job_finished && num_of_seconds_resets_take <= 40);

		if (!job_finished) {
			fprintf(stderr, "At least one module fails to come back! Please check the log.\n");
			SNGTC_ERROR("At least one module fails to come back! Please check the log.\n");
			return -1;
		} else {
			SNGTC_DEBUG("Resetting %i %s takes %u %s\n",
				detected, (detected > 1) ? "modules" : "module",
				num_of_seconds_resets_take, (num_of_seconds_resets_take == 1) ? "second" : "seconds");
		}
	}

	SNGTC_DEBUG("Detected %i(%i) Sangoma Transcoding Modules\n",
		   detected,sngtc_detected_vocallo_modules());


	err=sngtc_activate_modules(&g_init, &activated);
	if (err) {
		fprintf(stderr, "Failed to activate transcoding modules! Please check the logs.\n");
		SNGTC_ERROR("Failed to activate transcoding modules! Please check the logs.\n");
		return err;
	}

	if (activated != detected) {
		SNGTC_ERROR("Error: Detected %i modules but Activated %i\n",
			detected,activated);

		
		for (i=1;i<=detected;i++) {
			char fname[100];
			err=sngtc_module_get_instance(i,&vmod[i]);
			if (err) {
				continue;
			}
#if WIN32
			sprintf(fname,"sngtc-%s-ssi.bin",sngtc_module_mac(vmod[i]));
#else
			sprintf(fname,"/tmp/sngtc-%s-ssi.bin",sngtc_module_mac(vmod[i]));
#endif
			err=sngtc_module_read_sid(vmod[i],fname);
			if (err==0) {
				SNGTC_ERROR("%s: Sangoma Trancode retreived dump file: %s\n",
					sngtc_module_mac(vmod[i]),fname);
			}
		}

	}

	for (i=1;i<=detected;i++) {
	
		err=sngtc_module_get_instance(i,&vmod[i]);
		if (err) {
			continue;
		}

		err=sngtc_is_module_active(vmod[i]);
		if (err != 0) {
			SNGTC_ERROR("%s: Error: Sangoma Transcoding Module is Faulty!\n",
					   sngtc_module_mac(vmod[i]));
			continue;
		}
		
		err = sngtc_module_set_cpu_threshold(vmod[i], sngtc_cpu_high, sngtc_cpu_low);
		if (err) {
			continue;
		}

		SNGTC_INFO("%s: Setting cpu threshold Hi=%d/Lo=%d\n",
				sngtc_module_mac(vmod[i]), sngtc_cpu_high, sngtc_cpu_low);
	}

	
	launch_thread(&tid, vmod, monitor_vmod);
	
	SNGTC_DEBUG("Vocallo Detected=%i Activated=%i\n", detected, activated);

	return err;
}

void *monitor_vmod(void *vmod_id)
{
	uint16_t vmod = *(uint16_t*)vmod_id;

	SNGTC_INFO("Monitoring Sangoma Transcoding Modules\n");

	for (;;) {
		sngtc_poll();
		sngtc_mircoseconds_sleep(50000);
	}
	
	return NULL;
}

void *process_request(void *soap_obj)
{
	struct soap* soap = (struct soap*)soap_obj;

#if PLATFORM(Linux) 
	pthread_detach(sngtc_thread_id());
#endif

	for (;;) {
		//soap.socket_flags = MSG_NOSIGNAL; // use this
		//soap.accept_flags = SO_NOSIGPIPE; // or this to prevent sigpipe


		if (soap_serve(soap) != SOAP_OK) { // process RPC request
			//soap_print_fault(soap, stderr); // print error
			//soap_print_fault_location(soap,stderr);
			break;
		}

		SNGTC_DEBUG("request served\n");

		soap_destroy(soap); // clean up class instances
		soap_end(soap); // clean up everything and close socket
	}

	soap_destroy(soap); // clean up class instances
	soap_end(soap); // clean up everything and close socket
	soap_done(soap); // detach soap struct
	free(soap);

	sngtc_free_transcoding_sessions_by_tag(sngtc_thread_id());

	SNGTC_DEBUG("Thread Ended\n");
	return NULL;
}

#if PLATFORM(Windows) 
# include <windows.h>
static int win_sngtc_get_system_drive(char *out_buf)
{
	char cSystemWindowsDir[MAX_PATH];

	if(!GetSystemWindowsDirectory(cSystemWindowsDir, MAX_PATH)){
		return 1;
	}

	/* We are interested ONLY in the 1-st letter from something 
	 * like "C:\WINDOWS", which is "C". */
	sprintf(out_buf, "%c", cSystemWindowsDir[0]);

	return 0;
}
#endif

#if PLATFORM(Linux) 
static int do_ignore(int sig)
{
	sig=0;
   	return 0;
}

static int do_shut(int sig)
{
	SNGTC_ERROR("Caught SIG %d, Exiting Server!\n", sig);
	//sngtc_deactivate_modules();
	sleep(1);
	exit(0);
}
#endif

// -----------------------------------------------------------------------------

// the main program function

int main (int argc, char* argv[])
{
	int err;

	// -------------------------------------------------------
	
	// program parameters (0 means use default)
	
	// thread "types" are made up for this application, for the library there are
	// no distinctions
	
	// how many "type 1" threads to run
	uint nThreads1 = 1; //arg_u(argc, argv, "-threads1", 0);
	
	// how many "type 2" threads to run
	uint nThreads2 = 1; //arg_u(argc, argv, "-threads2", 0);
	
	// maximum message size supported
	uint maxMsgSize = 1024; //arg_u(argc, argv, "-msgsize", 0);

	memset(&g_init, 0, sizeof(g_init));
	g_init.log = log_printf;
	
	/* because the libsngtc library is not initialized yet, we must set the logger directly here */
	sngtc_log_func = log_printf;

#if PLATFORM(Linux) 
	(void) signal(SIGINT,(void *) do_shut);
	(void) signal(SIGTERM,(void *) do_shut);
	(void) signal(SIGPIPE,(void *) do_ignore);
	(void) signal(SIGUSR1,(void *) do_ignore);
	(void) signal(SIGHUP,(void *) do_shut);  
#endif

#if PLATFORM(Windows) 
	{
		char system_drive[10];

		if (win_sngtc_get_system_drive(system_drive)) {
			SNGTC_ERROR("Error: Failed to get System Drive!\n");
			return 1;
		}

		/* By default configuration is in 
	 	 * "SystemDrive:\sngtc\sngtc_server.conf.xml". */
		snprintf(cfg_file, sizeof(cfg_file), "%s:\\sngtc\\sngtc_server.conf.xml",
			system_drive);
	}
#endif

	// server TCP service port
#if 0
	uint servicePort = 11111; //arg_u(argc, argv, "-port", 0);
#endif
	
	// log level
	sngloglevel = arg_u(argc, argv, "-loglevel", SNGTC_LOGLEVEL_STATS);

	arg_str(argc, argv, "-conf", cfg_file,sizeof(cfg_file));

	// -------------------------------------------------------
	
	checkCommandLineOptions(argc, argv);


	printf("*\n* TS2 Libray Version: %s\n", server_getVersion());

#if 0
	printf("*\n* options:\n");
	printf("*    -loglevel: %d\n", logLevel);
	printf("*    -msgsize: %d\n",  maxMsgSize);
	printf("*    -port: %d\n",     servicePort);
	printf("*    -threads1: %d\n", nThreads1);
	printf("*    -threads2: %d\n", nThreads2);
#else
	printf("*\n* options:\n");
	printf("*    -loglevel: %s (%d)\n", TS2_LOGLEVEL_DECODE(logLevel),logLevel);
	printf("*    -conf    : %s\n", cfg_file);
#endif
	printf("*\n");

	SNGTC_DEBUG("Parsing XML configuration ...\n");
	err = sngtc_parse_xml_config_file(cfg_file);
	if (err) {
		SNGTC_ERROR("Error: Failed to parse %s\n", cfg_file);
		return err;
	}
	
	// phases of the application
	
	// phase 1: configuration (optional, otherwise uses default)
	
	SNGTC_DEBUG("setting the maximum message size\n");
	server_setMaxMessageSize(maxMsgSize);
	
#if 0
	SNGTC_DEBUG("setting the service port\n");
	server_setServicePort(servicePort);
#endif
	
	SNGTC_DEBUG("setting the log level\n");
	server_setLogLevel(logLevel);
	
	// phase 2: initialization
	SNGTC_DEBUG("initializing the server\n");
	server_init();
	
	// phase 3: add some threads
#if 0	
	SNGTC_DEBUG("adding %d threads os type 1\n", nThreads1);
	server_addThreads(1, sngtc_cli, "thread type 1\n");
#endif
	
	SNGTC_DEBUG("adding %d threads os type 2\n", nThreads2);
	server_addThreads(1, sngtc_soap_server, "thread type 2\n");
	// etc

	// phase 4: run!
	SNGTC_DEBUG("running the server!\n");
	server_run();

#if PLATFORM(Windows) 
	/* FIXME: the only way to exit the server is to "Ctrl-C" it!
	 Never reaching this point because server_run() never returns.
	 There must be a mechanism for graceful shutdown of the server for two cases:
	   1. running as Windows Service/Linux daemon
	   2. running as a command-line program */
	printf("Press any key to exit sngtc_server\n");
	_getch();
#endif
	// and that's all!
	exit(0);
}

// -----------------------------------------------------------------------------

/** Retrieves a program option.
*/

static uint arg_u(int argc, char* argv[], const char* argStr, uint def)
{
   int i;

   for (i = 1; i < argc - 1; i++)
   {
      if (!strcmp(argStr, argv[i])) // found option string (ex.: "-xxx")
      {
         if (!numeric(argv[i+1]))
         {
            printf("*\n* non-numeric characters in option %s: [%s]\n*\n",
               argv[i], argv[i+1]);
            usage();
         }
         return (uint)atoi(argv[i+1]);
      }
   }

   // didn't find argument, returns default value
   return def;
}

static uint arg_str(int argc, char* argv[], const char* argStr, char *str, int len)
{
   int i;

   for (i = 1; i < argc - 1; i++)
   {
      if (!strcmp(argStr, argv[i])) // found option string (ex.: "-xxx")
      {
         if (!strlen(argv[i+1]))
         {
            printf("*\n* non-string characters in option %s: [%s]\n*\n",
               argv[i], argv[i+1]);
            usage();
         }
		 strncpy(str,argv[i+1],len);
         return 0;
      }
   }

   // didn't find argument, returns default value
   return -1;
}

// -----------------------------------------------------------------------------

/** Checks if all command-line options are valid.
*/

static void checkCommandLineOptions(int argc, char** argv)
{
   int i;

   printf("*\n* checking command-line parameters\n");

   if (argc == 1)
   {
      printf("* no parameters!\n");
      return;
   }

   if (!(argc % 2))
   {
      printf("* probably missing value for a parameter!\n");
      usage();
   }

   for (i = 1; i < argc-1; i++)
   {
      int j;
      for (j = 0; options[j]; j++)
      {
          // printf("* checking: argv[%d]=[%s]\t\toptions[%d]=[%s]\n",
          //    i, argv[i], j, options[j]);

          if (!strcmp(argv[i], options[j])) // found option string (ex.: "-xxx")
          {
             printf("* ok, found %s\n", options[j]);
             // skips the option's value and goes to the next option
             i++;
             break;
          }
      } // for j

      if (!options[j])
      {
        // didn't find the option
         printf("*\n* unrecognized option: '%s'\n*\n", argv[i]);
         usage();
      }
   } // for i
}

// -----------------------------------------------------------------------------

/** Checks if a string is all numeric.
*/

static uint numeric(cchar* str)
{
   return (strspn(str, "0123456789") == strlen(str));
}

// -----------------------------------------------------------------------------

/** A typical thread to be run by the server framework.
*/

threadfunc sngtc_cli(void* arg)
{
   uint size, count = 0;
   char *bufIn, *bufOut;
   Message *msgIn, *msgOut;

   arg = 0; // only to avoid warnings...

   for (;;)
   {
      SNGTC_DEBUG("waiting for a message from a client\n");

      // waits for a message from a client
      msgIn = server_waitInputMessage();

      // ok, message received
      bufIn = server_messageBuffer(msgIn);
      size = server_messageSize(msgIn);

      // requests an (output) message to be sent as reply to the client
      msgOut = server_getFreeMessage();
      bufOut = server_messageBuffer(msgOut);

      // processes the (input) message received from the client
      server_printf("* message: length=%02d buf=[%.20s]\n", size, bufIn);
      // blah blah blah

      // creates the (output) message to be sent as reply to the client
      // (uses the same bytes and size, just so the client can check them)
      memcpy(bufOut, bufIn, size);
      server_setMessageSize(msgOut, size);

      if (!(++count % 10))
         SNGTC_DEBUG("%d messages processed now\n", count);

      // copies connection information from the input message to the output
      // message (this is needed so that the framework knows to which client
      // to send the output message)
      server_copyConnectionFromMessage(msgOut, msgIn);

      // releases the input message, it's not needed anymore
      // (this wouldn't be needed if the input message was reused)
      server_disposeMessage(msgIn);

      // makes the message available to be sent to the client
      server_dispatchOutputMessage(msgOut);
   }

   // that's it
   return 0;
}



// -----------------------------------------------------------------------------

/** Another thread to be run by the server framework.

    This thread provides a service that is unrelated to the framework, i.e,
    it does not use messages, connections, etc.
*/

threadfunc sngtc_soap_server(void* _parm)
{
	struct soap soap;
	SOAP_SOCKET sock_fd, s, i;// master and slave sockets
	int err=0;
	struct soap *tsoap=NULL;
	sngtc_thread_t tid;


	threads_running=1;

	memset(&soap,0,sizeof(soap));

	g_init.operation_mode = SNGTC_MODE_SOAP_SERVER;
	g_init.log = log_printf;
	g_init.create_rtp = NULL;
	g_init.create_rtp_port = NULL;
	g_init.destroy_rtp = NULL;


	//soap_init(&soap); // initialize the context (only once!)
	soap_init2(&soap, SOAP_IO_KEEPALIVE, SOAP_IO_KEEPALIVE);
	soap.max_keep_alive = -1; // at most 100 calls per keep-alive session
#if 0
	soap_init2(&soap, SOAP_IO_KEEPALIVE, SOAP_IO_KEEPALIVE);
	soap.max_keep_alive = 100; // at most 100 calls per keep-alive session
	soap.accept_timeout = 600; // optional: let server time out after ten minutes of inactivity
#endif

	soap.bind_flags |= SO_REUSEADDR;

	sock_fd = soap_bind(&soap, sngtc_server_ip_str , sngtc_server_port, 100);
	if (sock_fd < 0) {
		soap_print_fault(&soap, stderr);
		err=-1;
	} else {

#if 0
		int on = 1;
		err = setsockopt(sock_fd, SOL_SOCKET, SO_REUSEADDR, (void *)&on, sizeof(on));
		if (err) {
			SNGTC_ERROR("setsockopt(SO_REUSEADDR) failed");
			return err;
	}
#endif	
		err=vocallo_init();
		if (err) {
			err=-1;
			goto exit_error;
		}
		
//back_to_accept:

		SNGTC_INFO("Sangoma Transcoding Server Ready\n");
		SNGTC_DEBUG("Socket connection successful: master socket = %d\n", sock_fd);

		i = 0;
#if 1
		for (;;)
		{	
			s = soap_accept(&soap);
			if (!soap_valid_socket(s)) {
				if (soap.errnum) {
					soap_print_fault(&soap, stderr);
					err=-1;
					goto exit_error;
				}
				SNGTC_ERROR("server timed out\n");
				break;
			}
			SNGTC_DEBUG("Thread %d accepts socket %d connection from IP %d.%d.%d.%d\n", i, s,
						(soap.ip >> 24)&0xFF, (soap.ip >> 16)&0xFF, (soap.ip >> 8)&0xFF, soap.ip&0xFF);
			i++;

			tsoap = soap_copy(&soap); // make a safe copy
			if (!tsoap)
				break;

			launch_thread(&tid, tsoap, process_request);
			
		}

#else 
		s = soap_accept(&soap);
		if (s < 0)
		{
			soap_print_fault(&soap, stderr);
			goto exit_error;
		}
		SNGTC_WARN("%d: accepted connection from IP=%d.%d.%d.%d socket=%d\n\n", i,
					(soap.ip >> 24)&0xFF, (soap.ip >> 16)&0xFF, (soap.ip >> 8)&0xFF, soap.ip&0xFF, s);
	
		for (i = 1; ; i++)
		{
			//soap.socket_flags = MSG_NOSIGNAL; // use this
			//soap.accept_flags = SO_NOSIGPIPE; // or this to prevent sigpipe
			
					
			if (soap_serve(&soap) != SOAP_OK) { // process RPC request
				soap_print_fault(&soap, stderr); // print error
				soap_print_fault_location(&soap,stderr);
				break;
			}
		
			fprintf(stderr, "request served\n");
		
			soap_destroy(&soap); // clean up class instances
			soap_end(&soap); // clean up everything and close socket
		}
			
		goto back_to_accept;
#endif
	}

exit_error:
			
	soap_destroy(&soap); // clean up class instances
	soap_end(&soap); // clean up everything and close socket
	
	soap_done(&soap); // close master socket and detach context 

	threads_running=0;

	SNGTC_ERROR("Exiting SOAP Thread err=%i\n",err);

	/* Exit applicaton */
	exit(err);
	
}


int ns1__sngtc_create_transcoding_session(struct soap *soap, struct ns1__sngtc_codec_request *ns_codec_req, struct ns1__sngtc_codec_reply *ns_codec_reply)
{
	int err;
	struct sngtc_codec_request codec_req;
	struct sngtc_codec_reply codec_reply;

	soap->error=SOAP_OK;
	
	if (sizeof(*ns_codec_reply) != sizeof(codec_reply)) {
		SNGTC_ERROR("%s: Internal error ns_codec_reply=%i != codec_reply=%i\n",
					__FUNCTION__,sizeof(*ns_codec_reply),sizeof(codec_reply));
		soap->error=SOAP_USER_ERROR;
		goto exit_session;
	}
	
	if (sizeof(*ns_codec_req) != sizeof(codec_req)) {
		SNGTC_ERROR("%s: Internal error ns_codec_req=%i != codec_req=%i\n",
					__FUNCTION__,sizeof(*ns_codec_req),sizeof(codec_req));
		soap->error=SOAP_USER_ERROR;
		goto exit_session;
	}
	
	SNGTC_DEBUG("Calling %s\n",__FUNCTION__);

	memcpy(&codec_req, ns_codec_req, sizeof(codec_req));
	memcpy(&codec_reply, ns_codec_reply, sizeof(codec_reply));
	
	codec_req.tag = (uint64_t)sngtc_thread_id();
	codec_req.usr_priv=NULL;

	/* No need to check return code becuase
	   the codec_reply will pass the return code to the
	   client */
	err=sngtc_create_transcoding_session(&codec_req, &codec_reply, 0);
	if (err) {
		SNGTC_ERROR("sng_create_transcoding_session failed Result=%i\n",codec_reply.result);
	}

	/* We are not using SOAP error codes here. Error code is in codec_reply  */
	
	SNGTC_DEBUG("sng_create_transcoding_session session 0x%08X\n",codec_reply.codec_rtp_session_idx);
	
	memcpy(ns_codec_reply, &codec_reply, sizeof(codec_reply));

exit_session:
	
	return soap->error;
}

int ns1__sngtc_free_transcoding_session(struct soap *soap, struct ns1__sngtc_codec_reply *ns_codec_reply, int *result)
{
	int err;
	struct sngtc_codec_reply codec_reply;

	*result=0;
	
	SNGTC_DEBUG("Calling %s id=0x%08X \n",__FUNCTION__,
		   ns_codec_reply->codec_rtp_session_idx);
	
	memcpy(&codec_reply, ns_codec_reply, sizeof(codec_reply));
	
	err=sngtc_free_transcoding_session(&codec_reply);
	if (err) {
		SNGTC_ERROR("sng_free_transcoding_session failed\n");
		*result=codec_reply.result;
	}

	soap->error=SOAP_OK;
	
	return SOAP_OK;
}



// -----------------------------------------------------------------------------

/** Directions for using this program.
*/

static void usage(void)
{
   printf("* Program usage:\n*\n");
   printf("*    sngtc_server\n"
      "*           [-conf <file location>]\n"
      "*           [-loglevel <value>]\n");

   exit(1);
}

// -----------------------------------------------------------------------------
// the end
