/// @source      MessageQueue.c
/// @description Implementation of class MessageQueue.
//  See licensing information in the file README.TXT.

// -----------------------------------------------------------------------------

// includes

// common configuration options & declarations (always include first)
#include "config.h"

// C language includes
#include <assert.h>
#include <stdlib.h>  // calloc

// framework includes
#include "Message.h"      /* messager_xxx functions     */
#include "MessageQueue.h" /* messageQueue_xxx functions */
#include "util/Queue.h"   /* queue_xxx functions        */
#include "util/Log.h"     /* log_xxx functions          */
#include "util/Mutex.h"   /* mutex_xxx functions        */

// -----------------------------------------------------------------------------

// global declarations

/// For identification in the log file
/// @private @memberof MessageQueue
static cchar sourceID[] = "ZMQ";

/// MessageQueue data structure.
/// A message queue that is owned by the QueueManager. A message queue collects
/// @link Message messages @endlink with a specific use within the framework.
/// <b>(1)</b> The free  message queue contains the messages not presently in
/// use, available to be claimed by other entities within the framework.
/// <b>(2)</b> The output message queue contains messages generated by the
/// application @link Thread threads @endlink, that are scheduled to be sent to
/// the client applications; the actual transfer process for an output message
/// begins when it's removed from the output messsage queue by the
/// ConnectionManager, and its ownership is transferred to a Connection object.
/// <b>(3)</b> The input message queue contains messages already received from
/// the client applications by the framework, waiting to be processed by an
/// application Thread.
struct MessageQueue
{
   Queue* messages; ///< @private MessageQueue, implemented
   mutex_t*  mutex; ///< @private Contention mutex for the MessageQueue
};

// helper functions
/// @cond hides_from_doxygen
static uint checkMessageFlags(void*, void*);
/// @endcond

// -----------------------------------------------------------------------------
// PUBLIC INTERFACE
// -----------------------------------------------------------------------------

/** Adds a Message to this MessageQueue.

    @param
    messageQueue : the MessageQueue which is being operated upon

    @param [in]
    message : the Message that is being added to the MessageQueue

    @memberof MessageQueue
*/

TS2API void messageQueue_add(MessageQueue* messageQueue, const Message* message)
{
   log_func(messageQueue_add);
   log_fdebug("adding a message to a message queue");

   assert(messageQueue);

   mutex_lock(messageQueue->mutex);
   queue_addTail(messageQueue->messages, message);
   mutex_unlock(messageQueue->mutex);
}

// -----------------------------------------------------------------------------
// PUBLIC INTERFACE
// -----------------------------------------------------------------------------

/** Queries the number of @link Message Messages @endlink in this MessageQueue.

    @param
    messageQueue : the MessageQueue which is being operated upon

    @return
    number of message in the MessageQueue

    @memberof MessageQueue
*/

TS2API uint messageQueue_count(const MessageQueue* messageQueue)
{
   assert(messageQueue);
   return queue_size(messageQueue->messages);
}

// -----------------------------------------------------------------------------
// PUBLIC INTERFACE
// -----------------------------------------------------------------------------

/** Creates a MessageQueue.

    @param [in]
    nMessages : number of @link Message Messages @endlink to be added to the new
    MessageQueue

    @param [in]
    flags : flags to be applied to the @link Message Messages @endlink that will
    be added to the new MessageQueue

    @return
    the new MessageQueue

    @memberof MessageQueue
*/

TS2API MessageQueue* messageQueue_create(uint nMessages, ushort flags)
{
   Message* message;
   MessageQueue* messageQueue;

   log_func(messageQueue_create);
   log_finfo("creating a message queue with %d messages and flags %04x",
      nMessages, flags);

   messageQueue = (MessageQueue*) calloc(1, sizeof(messageQueue));
   assert(messageQueue);

   messageQueue->messages = queue_create(100); // configurable ???

   while (nMessages--)
   {
      message = message_create(flags);
      queue_addTail(messageQueue->messages, message);
   }

   log_finfo("creating the message queue contention mutex");
   messageQueue->mutex = mutex_create();

   return messageQueue;
}

// -----------------------------------------------------------------------------

/** Removes and destroys @link Message Messages @endlink with specific lifetime
    from this MessageQueue.

    @param
    messageQueue : the MessageQueue which will be operated upon

    @param [in]
    nMessages : number of @link Message Messages @endlink that will be destroyed

    @param [in]
    flags : lifetime flags of the @link Message Messages @endlink that will be
    destroyed

    @return
    number of @link Message Messages @endlink destroyed

    @memberof MessageQueue
*/

TS2API uint messageQueue_destroyMessages(MessageQueue* messageQueue, uint nMessages,
   ushort flags)
{
   Message* message;
   uint nDestroyed = 0;

   log_func(messageQueue_destroyMessages);
   log_finfo("destroying %d messages with flag %04X", nMessages, flags);

   assert(messageQueue);

   log_finfo("locking the mutex of the message queue");
   mutex_lock(messageQueue->mutex);

   for (;;)
   {
      // all messages already destroyed ?
      if (!nMessages--)
      {
         log_fdebug("all the requested messages were destroyed");
         break;
      }

      log_finfo("removing a message from a message queue");
      message = (Message*)queue_remove(messageQueue->messages, checkMessageFlags,
         &flags);

      // no message destroyed!
      if (!message)
      {
         log_fwarn("couldn't destroy a message!");
         break;
      }

      log_finfo("destroying message %04X", message);
      message_destroy(message);
      nDestroyed++;
   }

   log_finfo("unlocking the mutex of the message queue");
   mutex_unlock(messageQueue->mutex);

   log_finfo("%d messages with flags 0x%04X were destroyed", nDestroyed, flags);

   return nDestroyed;
}

// -----------------------------------------------------------------------------
// PUBLIC INTERFACE
// -----------------------------------------------------------------------------

/** Removes the first Message from this MessageQueue.

    @param
    messageQueue : the MessageQueue which will be operated upon

    @return
    address of @link Message Message @endlink just removed (NULL if MessageQueue
    empty)

    @memberof MessageQueue
*/

TS2API Message* messageQueue_get(MessageQueue* messageQueue)
{
   Message* message;

   log_func(messageQueue_get);
   log_fdebug("retrieving a message from a message queue");

   assert(messageQueue);

   mutex_lock(messageQueue->mutex);
   message = queue_removeHead(messageQueue->messages);
   mutex_unlock(messageQueue->mutex);

   return message;
}

// -----------------------------------------------------------------------------
// PUBLIC INTERFACE
// -----------------------------------------------------------------------------

/** Checks to see if this MessageQueue is empty.

    @param
    messageQueue : the MessageQueue which will be operated upon

    @return
    true if empty, false otherwise.

    @memberof MessageQueue
*/

TS2API uint messageQueue_isEmpty(const MessageQueue* messageQueue)
{
   assert(messageQueue);
   return !queue_size(messageQueue->messages);
}

// -----------------------------------------------------------------------------
// INTERNAL HELPER FUNCTIONS
// -----------------------------------------------------------------------------

/** Checks to see if a Message has the specified flag bits set.

    @param
    _message : the Message which will be checked

    @param [in]
    _flags : address of variable with the flag bits to be checked

    @return
    true if all bits set, false otherwise.

    @private @memberof MessageQueue
*/

static uint checkMessageFlags(void* _message, void* _flags)
{
   uint result;
   Message* message = (Message*)_message;
   ushort flags = *(ushort*)_flags;

// log_func(checkMessageFlags);

   assert(message);

   result = ((message_flags(message) & flags) == flags);

// log_fdebug("result:%d message:%X message_flags: %02X flags:%02X", result,
//    message, message_flags(message), flags);

   return result;
}

// -----------------------------------------------------------------------------
// the end
