/*******************************************************************************//**
 * \file sng_tc_if.h
 * \brief Sangoma Transcoding API Interface File
 * \brief Defines Sangoma Transcoding API functions and structures
 *              
 * Sangoma Transcoder Library
 *
 * Author(s): Nenad Corbic <ncorbic@sangoma.com>
 *            Moises Silva <moy@sangoma.com>
 *
 * Copyright (C) 2010 Sangoma Technologies
 *
 * Redistribution and use is permitted only in binary forms without
 * modification and provided that the following conditions are met:
 *
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 *
 * * Neither the name of the original author; nor the names of any contributors
 *   may be used to endorse or promote products derived from this software
 *   without specific prior written permission.
 *
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *
 */


#ifndef __LIB_SANGOMA_TRANSCODE_IF__
#define __LIB_SANGOMA_TRANSCODE_IF__

#include "sng_tc_os.h"

/*!
 * \enum sngtc_codec_definition
 * \brief sngtc codec definition, used to identify a desired codec.
 */
enum sngtc_codec_definition {
	SNGTC_CODEC_MIN_ID	= 0,

	SNGTC_CODEC_PCMU	= 10,
	SNGTC_CODEC_PCMA	= 20,
	SNGTC_CODEC_L16_1	= 30,
	SNGTC_CODEC_L16_2	= 40,
	SNGTC_CODEC_G729AB	= 50,
	SNGTC_CODEC_G722	= 60,
	SNGTC_CODEC_GSM_FR	= 70,
	SNGTC_CODEC_GSM_EFR	= 80,

	SNGTC_CODEC_AMR		= 90,
	SNGTC_CODEC_AMR_475     = 90,
	SNGTC_CODEC_AMR_515     = 91,
	SNGTC_CODEC_AMR_590     = 92,
	SNGTC_CODEC_AMR_670     = 93,
	SNGTC_CODEC_AMR_740     = 94,
	SNGTC_CODEC_AMR_795     = 95,
	SNGTC_CODEC_AMR_1020    = 96,
	SNGTC_CODEC_AMR_1220    = 97,
	SNGTC_CODEC_AMR_WB	= 100,

	SNGTC_CODEC_G726_16	= 110,
	SNGTC_CODEC_G726_24	= 120,
	SNGTC_CODEC_G726_32	= 130,
	SNGTC_CODEC_G726_40	= 140,

	SNGTC_CODEC_ILBC	= 150,
	SNGTC_CODEC_ILBC_133	= 150,
	SNGTC_CODEC_ILBC_152	= 160,

	SNGTC_CODEC_G723_1      = 170,
	SNGTC_CODEC_G723_1_53   = 170,
	SNGTC_CODEC_G723_1_63   = 180,

	SNGTC_CODEC_SIREN7_24   = 190,
	SNGTC_CODEC_SIREN7_32   = 191,
	SNGTC_CODEC_SIREN7_48   = 192,

	SNGTC_CODEC_MAX_ID 	= 255,
};

/*!
 * \def SNGTC_CODEC_ID_DECODE
 * \brief Convert codec id enum to string 
 */
#define SNGTC_CODEC_ID_DECODE(id) \
		(id==SNGTC_CODEC_PCMU) ? "SNGTC_CODEC_PCMU" : \
		(id==SNGTC_CODEC_PCMA) ? "SNGTC_CODEC_PCMA" : \
		(id==SNGTC_CODEC_L16_1) ? "SNGTC_CODEC_L16_1" : \
		(id==SNGTC_CODEC_L16_2) ? "SNGTC_CODEC_L16_2" : \
		(id==SNGTC_CODEC_G729AB) ? "SNGTC_CODEC_G729AB" : \
		(id==SNGTC_CODEC_G722) ? "SNGTC_CODEC_G722" : \
		(id==SNGTC_CODEC_GSM_FR) ? "SNGTC_CODEC_GSM_FR" : \
		(id==SNGTC_CODEC_GSM_EFR) ? "SNGTC_CODEC_GSM_EFR" : \
		(id==SNGTC_CODEC_AMR_475) ? "SNGTC_CODEC_AMR_475" : \
		(id==SNGTC_CODEC_AMR_515) ? "SNGTC_CODEC_AMR_515" : \
		(id==SNGTC_CODEC_AMR_590) ? "SNGTC_CODEC_AMR_590" : \
		(id==SNGTC_CODEC_AMR_670) ? "SNGTC_CODEC_AMR_670" : \
		(id==SNGTC_CODEC_AMR_740) ? "SNGTC_CODEC_AMR_740" : \
		(id==SNGTC_CODEC_AMR_795) ? "SNGTC_CODEC_AMR_795" : \
		(id==SNGTC_CODEC_AMR_1020) ? "SNGTC_CODEC_AMR_1020" : \
		(id==SNGTC_CODEC_AMR_1220) ? "SNGTC_CODEC_AMR_1220" : \
		(id==SNGTC_CODEC_AMR_WB) ? "SNGTC_CODEC_AMR_WB" : \
		(id==SNGTC_CODEC_G726_16) ? "SNGTC_CODEC_G726_16" : \
		(id==SNGTC_CODEC_G726_24) ? "SNGTC_CODEC_G726_24" : \
		(id==SNGTC_CODEC_G726_32) ? "SNGTC_CODEC_G726_32" : \
		(id==SNGTC_CODEC_G726_40) ? "SNGTC_CODEC_G726_40" : \
		(id==SNGTC_CODEC_ILBC_133) ? "SNGTC_CODEC_ILBC_133" : \
		(id==SNGTC_CODEC_ILBC_152) ? "SNGTC_CODEC_ILBC_152" : \
		(id==SNGTC_CODEC_G723_1_53) ? "SNGTC_CODEC_G723_1_53" : \
		(id==SNGTC_CODEC_G723_1_63) ? "SNGTC_CODEC_G723_1_63" : \
		(id==SNGTC_CODEC_SIREN7_24) ? "SNGTC_CODEC_SIREN7_24" : \
		(id==SNGTC_CODEC_SIREN7_32) ? "SNGTC_CODEC_SIREN7_32" : \
		(id==SNGTC_CODEC_SIREN7_48) ? "SNGTC_CODEC_SIREN7_48" : "Unknown"

/*!
 * \enum sngtc_error_codes
 * \brief Error codes used by the library
 */
enum sngtc_error_codes {
	SNTC_OK					=0,
	SNTC_ERROR,
	SNTC_ERROR_MEM_ALLOC,
	SNTC_ERROR_ALL_CHANNELS_BUSY,
	SNTC_ERROR_INTERNAL_LOGIC_ERROR,
	SNTC_ERROR_FAILED_TO_CREATE_RTP_SESSION,
	SNTC_ERROR_FAILED_TO_CONNECT_TERMS,
	SNTC_ERROR_INVALID_CODEC_REQ,
	SNTC_ERROR_INVALID_CODEC_ID_A_IN_CODEC_REQ,
	SNTC_ERROR_INVALID_CODEC_ID_B_IN_CODEC_REQ,
	SNTC_ERROR_INVALID_CODEC_ID_TO_IANA,
	SNTC_ERROR_CREATE_RTP_PORT,
	SNTC_ERROR_CREATE_RTP,
};

/*!
  \struct sngtc_codec_request_leg
  \brief Used to configure codec information for each leg of the call.

  \typedef sngtc_codec_request_leg_t
*/
typedef struct sngtc_codec_request_leg
{
	uint32_t 	codec_id;		/*!< codec id: enum sngtc_codec_definition */
	uint32_t	ms;             /*!< codec ms parameter in mmilliseconds */

	uint32_t	host_ip;        /*!< ip address of a host */
	uint32_t	host_netmask;   /*!< ip address of a host netmask */
	uint32_t	host_udp_port;  /*!< host rtp port */

} sngtc_codec_request_leg_t;


/*!
  \struct sngtc_codec_request
  \brief Used to configure the whole call session that includes 2 legs of the call.

  \typedef sngtc_codec_request_t
*/

typedef struct sngtc_codec_request
{
	void 		*usr_priv;			/*!< user void pointer used in user callbacks */
	uint64_t	tag;                /*!< identifies a session with a tag. Used to shutdown all sessions via tag */
                                    /*!< enable or disable rtcp in sangtc */
	uint8_t		rtcp_enable;

	struct sngtc_codec_request_leg a;  /*!< leg a configuration */
	struct sngtc_codec_request_leg b;  /*!< leg b configuration */

} sngtc_codec_request_t;



/*!
  \struct sngtc_codec_reply_leg
  \brief Per leg codec configuration obtained from the sangoma transcoding module as a resopnse to the codec request.

  \typedef sngtc_codec_reply_leg_t
*/
typedef struct sngtc_codec_reply_leg
{
	uint32_t	codec_ip;  		/*!< ip address of sngtc rtp session */
	uint32_t	codec_netmask;  /*!< netmask of sngtc rtp session */
	uint32_t	codec_udp_port; /*!< rtp port of the sngtc rtp session */

	uint32_t	host_ip;        /*!< host ip info obtained from codec_request structure */
	uint32_t	host_netmask;   /*!< host netmask info obtained form codec_request structure */
	uint32_t	host_udp_port;  /*!< host rtp port obtained form codec_request structure */
                                
	uint32_t	iana_code;      /*!< iana code used in the rtp header: payload type */

}sngtc_codec_reply_leg_t;




/*!
  \struct sngtc_codec_reply
  \brief Codec cfg from the sngtc module for the whole session. 2 legs. as the response codec request.

  \typedef sngtc_codec_reply_t
*/
typedef struct sngtc_codec_reply
{
	uint32_t 	codec_module_session_idx;  /*!< id to identify sngtc module */
	uint32_t	codec_rtp_session_idx;     /*!< id to identify rtp session inside the sngtc module */

	struct sngtc_codec_reply_leg a;       /*!< per leg sngtc codec information */
	struct sngtc_codec_reply_leg b;       /*!< per leg sngtc codec information */

	void		*tx_fd;                   /*!< For library use: used during user callbacks */
	uint32_t	tx_iana;                  /*!< For library use: used during user callbacks */

	void		*rx_fd;                   /*!< For library use: used during user callbacks */
	uint32_t	rx_iana;                  /*!< For library use: used during user callbacks */

	int32_t		result;					  /*!< used in soap mode to provide return code */

} sngtc_codec_reply_t;



#define SNGTC_MAX_MAC_RANGE 		32
#define SNGTC_MAX_HOST_VOCALLO_NIC 	32
#define SNGTC_MAX_ETH_SZ			6

/*!
  \struct sngtc_eth_mac
  \brief Ethernet mac structure contains hex and string representation so of hwaddr.

  \typedef sngtc_eth_mac_t
 */
typedef struct sngtc_eth_mac
{
	char mac_str[18];
	uint8_t mac[6];
} sngtc_eth_mac_t;

/*!
  \struct sngtc_ip
  \brief IPv4 Address structure contains hex and string representation so of ip address.

  \typedef sngtc_ip_t
 */
typedef struct sngtc_ip
{
	char ip_str[25];
	uint32_t ip;
}sngtc_ip_t;

/*!
  \struct system_eth_cfg
  \brief Ethernet interface structure, reprensent system info for a ethernet interface.

  \typedef system_eth_cfg_t
 */
typedef struct system_eth_cfg{
	int 			init;
	char 			name[255];
	sngtc_eth_mac_t mac;
	sngtc_ip_t 		ip;
	sngtc_ip_t 		mask;
	uint32_t 		flags;
	uint8_t 		set_if_up;
	sngtc_ip_t		gateway;
}system_eth_cfg_t;



/*!
  \struct sngtc_host_nic_vocallo_cfg
  \brief Sangoma transcode init structure used to configure the transcode with initial parmeters.
  \typedef sngtc_host_vocallo_cfg
*/
typedef struct sngtc_host_nic_vocallo_cfg
{
	uint32_t 	vocallo_base_udp_port;
	uint32_t	vocallo_ip;
	uint32_t	vocallo_ip_netmask;

	uint32_t	host_ip;
	uint32_t	host_ip_netmask;

	uint32_t	gateway_ip;
	
	sngtc_eth_mac_t	host_mac;

	/* Optional to select a speicifc vocallo */
	sngtc_eth_mac_t	vocallo_macs[SNGTC_MAX_MAC_RANGE];
	uint32_t		vocallo_macs_sz;

	uint8_t		silence_suppression;

} sngtc_host_nic_vocallo_cfg_t;


typedef int (*sngtc_log_func_t)( int level, char *fmt, ...);

/*!
  \struct sngtc_init_cfg
  \brief Sangoma transcode init structure used to configure the transcode with initial parmeters.
  \typedef sngtc_init_cfg_t
*/ 

typedef struct sngtc_init_cfg 
{
	/* Library operation mode: LIBRARY, SOAP_CLIENT, SOAP_SERVER */
	int								operation_mode;

	/* Host NIC configuration is only used in non soap_mode.  List of structures 
	   each indicating the ethernet interface information that should be used 
	   to probe for sangoma transcoding cards.  */
	uint32_t						host_nic_vocallo_sz;
	sngtc_host_nic_vocallo_cfg_t	host_nic_vocallo_cfg[SNGTC_MAX_HOST_VOCALLO_NIC];

	/* log function is used when library would like to print information.  
	   Each log comes with a log level for easy priority parsing */
	sngtc_log_func_t log;

	/* create rtp callback is used during create transcoding session call.  
	   As soon as there is enough information obtained by the library for each leg of the call.  
	   The create_rtp is called so that application can create its rtp session and start 
	   transmitting and receiving data */ 
	int (*create_rtp)(void * usr_priv, sngtc_codec_request_leg_t *codec_reg_leg, sngtc_codec_reply_leg_t* codec_reply_leg, void **rtp_fd);
	
	/* create_rtp_port is called during the create transcoding session.  In some applications 
	   the rtp port that will be used to create rtp session must be obtained earlier.  
	   The host_ip is the ip address of the host application, which can be used in 
	   rtp hunting procedures. */
	int (*create_rtp_port)(void * usr_priv, uint32_t host_ip, uint32_t *rtp_port,  void **rtp_fd);

	/* destroy_rtp callback is called during free transcoding session call.  
	   At this time host application is able to free its rtp session resources */
	int (*destroy_rtp)(void * usr_priv, void *fd);

	/* release_rtp_port callback is called only when creating a transcoding session fails
	   before the create_rtp callback was called and therefore only the port needs to be released.
	   If the create_rtp callback was already called, only destroy_rtp will be called and this callback not. */
	int (*release_rtp_port)(void * usr_priv, uint32_t host_ip, uint32_t rtp_port,  void *rtp_fd);

} sngtc_init_cfg_t;

/*!
   \fn  sngtc_detect_init_modules(sngtc_init_cfg_t *cfg, int *detected)
   \brief Detect and group sngtc modules on specified lan interfaces. 
   \param sngtc_init_cfg_t  contains the list of eth interfaces that should be probed for sangoma transcoding cards.
   \param detected returns the number of detected modules
   \return  0 ok ,  non-zero error.

   Detect and group found sangoma transcoding modules.n ethernet broadcast 
   is sent out over the LAN. If a sangoma transcoding module is active it will respond. 
   Each sngtc module detected is put in resource pool to be used by the application.
   
   sngtc_init_cfg_t *cfg cotains the list of ethernet intefaces that should b
   e probed for sangoma transcoding cards. 
   
   int *detected returns the number of detected modules
   
   Return code:  0 ok ,  non-zero error.
*/

int _SNGTC_API sngtc_detect_init_modules(sngtc_init_cfg_t *cfg, int *detected);

/*! 

   \fn  int sngtc_activate_modules(sngtc_init_cfg_t *cfg, int *activated)   
   \brief Iterate over all detected modules and perform a full reset on each module.
   \param sngtc_init_cfg_t  Same cfg strucutre used in detect_init fuction.
   \param activated returns the number of activated modules
   \return  0 ok ,  non-zero error.

   Iterate over all detected modules and perform a full reset on each module. Check that firmware 
   and configuration of each module is correct. Return number of successfully activated sangoma 
   transcoding modules.
   
   Arguments:
   sngtc_init_cfg_t *cfg Initial configuration structure. 
                         Not really used by activate function, but left for future use.  
						 
   int *activatedReturns number of activated modules.Return code:0 ok,  non-zero error
*/
int _SNGTC_API sngtc_activate_modules(sngtc_init_cfg_t *cfg, int *activated);


/*! 
   \fn  int sngtc_deactivate_modules(void)
   \brief Free all active and detected modules and release resources. 
   \return  0 ok ,  non-zero error.
*/
int _SNGTC_API sngtc_deactivate_modules(void);

/*! 
 * \fn int sngtc_create_transcoding_session (struct sngtc_codec_request *codec_req, struct sngtc_codec_reply *codec_reply, int start_module)
 * \brief Create transcoding session. Each transcoding session creates a 2 RTP legs.
 * \param codec_req contains codec information for each leg as well as host ip and port information.
 * \param codec_reply structure is filled by the sangoma transcoding module and contains the IP 
 * \param start_module can be used to select a specific sangoma transcoding module.
 * \return  0 ok , non-zero failed.
 * 
 * Create transcoding session. Each transcoding session creates a 2 RTP legs.
 * The codec_req structure contains codec information for each leg as well as host ip and port information.
 * Once the codec legs are created, the sangoma transcode module will start sending RTP
 * packets to the IP address provided by the codec_req structure.  The host is expected to
 * create RTP sessions that will be listening to the incoming rtp traffic.
 *
 * The codec_reply structure is filled by the sangoma transcoding module and contains the IP
 * and port information that will be used by the host to transmit RTP data to sangoma transcoding module.
 *
 * start_module can be used to select a specific sangoma transcoding module.
 * By default the value of start_module is ZERO, indicating auto select.

*/
int _SNGTC_API sngtc_create_transcoding_session (struct sngtc_codec_request *codec_req, struct sngtc_codec_reply *codec_reply, int start_module);


/*!
 *\fn int sngtc_free_transcoding_session   (struct sngtc_codec_reply *codec_reply)
 *\brief Free transcoding session. The RTP data from sangoma transcoding card will stop.
 *\param codec_reply must be the same structure that was received from sngtc_create_transcoding_session
 *\return 0 - ok, non-zero failed
 * 
 * Free transcoding session. The RTP data from sangoma transcoding card will stop
 * and resources on the sangoma transcoding card will be freed related to that transcoding channel.
 *
 * codec_reply structure must be the same structure that was received from
 * sangoma transcoding module on session create.  It contains the session id that is
 * used by the library to indicate what session to free.
*/
int _SNGTC_API sngtc_free_transcoding_session   (struct sngtc_codec_reply *codec_reply);

/*!
 *\fn int sngtc_detected_vocallo_modules(void)
 *\brief Get number of detected sangoma transcoding modules already detected.
 *\return 0 or greater:number of modules detecte, non-zero: failed
 *
 */
int _SNGTC_API sngtc_detected_vocallo_modules(void);

/*!
 *\fn char *sngtc_module_mac(uint16_t module_idx)
 *\brief Get module id based on module index value.  Module id is its MAC address in string format.\
 *\param module_idx unique id of detected sangoma transcoding module.
 *\return string of module MAC address.
 *
 */
_SNGTC_API char * sngtc_module_mac(uint16_t module_idx);

#endif

