/* ---------------------------------------------------------------------------
          Copyright (c) 2003-2008 Micrel, Inc.  All rights reserved.
   ---------------------------------------------------------------------------

    NdisDriver.c - NDIS driver functions.

    Author      Date        Description
    THa         12/01/03    Created file.
    THa         06/27/05    Updated for version 0.1.5.
    THa         09/29/05    Changed descriptor structure.
    THa         02/23/06    Updated for WinCE 5.0.
    THa         03/14/06    Release 1.0.0.
    THa         06/02/06    Report statistics from MIB counters.
    THa         07/27/06    Version 1.4 supports NDIS 5.
    THa         10/04/06    Version 1.6 delays reporting driver reset status.
    THa         12/21/06    Version 1.7 fixes MIB counter read problem.
    THa         12/18/07    Add NDIS 5.1 support.
    THa         01/22/08    Update for 64-bit Windows.
    THa         02/15/08    Add force link support.
    THa         12/15/08    Fix getting SpeedDuplex parameter problem.
    DAVIDCAI    12/21/2009  Change the DRV_VERSION to 1.0.11. The new driver works for new 
    												Micrel LinkMD driver and can pass back link status for each port. 
    ---------------------------------------------------------------------------
*/


#include "target.h"
#include "hardware.h"
#include "NdisDevice.h"
#include "NdisOid.h"


/* -------------------------------------------------------------------------- */

#if defined( DEF_KS8842 )
#define DRV_NAME     "KSZ8842"
#elif defined( DEF_KS8841 )
#define DRV_NAME     "KSZ8841"
#else
#error Either DEF_KS8841 or DEF_KS8842 has to be defined.
#endif
#define BUS_NAME     "PCI"
#define DRV_VERSION  "1.0.11"
#define DRV_RELDATE  "Dec 8, 2009"

#ifdef DBG
static char version[] =
    "Micrel " DRV_NAME " " BUS_NAME " " DRV_VERSION " (" DRV_RELDATE ")";
#endif

/* -------------------------------------------------------------------------- */

#ifdef UNDER_CE

#if DBG
DBGPARAM dpCurSettings = {
    TEXT( NDIS_NAME ),
    {
        TEXT( "Error" ),
        TEXT( "Warning" ),
        TEXT( "Func" ),
        TEXT( "Init" ),
        TEXT( "Intr" ),
        TEXT( "Rx" ),
        TEXT( "Tx" ),
        TEXT( "Link" ),
        TEXT( "Undefined" ),
        TEXT( "Undefined" ),
        TEXT( "Undefined" ),
        TEXT( "Undefined" ),
        TEXT( "Undefined" ),
        TEXT( "Undefined" ),
        TEXT( "Undefined" ),
        TEXT( "Undefined" )
    },
    ZONE_INIT_MASK | ZONE_ERROR_MASK
};
#endif


BOOL __stdcall
DllEntry (
    HANDLE hDLL,
    DWORD  dwReason,
    LPVOID lpReserved )
{
    switch ( dwReason )
    {
        case DLL_PROCESS_ATTACH:
            DEBUGREGISTER( hDLL );
            DEBUGMSG( ZONE_INIT, ( TEXT( "KS884X: DLL_PROCESS_ATTACH\r\n" )));
            DisableThreadLibraryCalls(( HMODULE ) hDLL );
            break;
        case DLL_PROCESS_DETACH:
            DEBUGMSG( ZONE_INIT, ( TEXT( "KS884X: DLL_PROCESS_DETACH\r\n" )));
            break;
    }
    return TRUE;
}

/* ------------------------------------------------------------------------- */

#define MIN_DETECT_KEY  40
#define MAX_DETECT_KEY  99


typedef struct _REG_VALUE_DESCR {
    LPWSTR val_name;
    DWORD  val_type;
    PBYTE  val_data;
} REG_VALUE_DESCR, *PREG_VALUE_DESCR;


// Values for [HKEY_LOCAL_MACHINE\Drivers\PCMCIA\Detect\40]
REG_VALUE_DESCR DetectKeyValues[] = {
    ( TEXT( "Dll" )), REG_SZ, ( PBYTE )( TEXT( "NdisDriver.DLL" )),
    ( TEXT( "Entry" )), REG_SZ, ( PBYTE )( TEXT( "DetectNdisDevice" )),
    NULL, 0, NULL
};

// Values for [HKEY_LOCAL_MACHINE\Drivers\PCMCIA\NdisDriver]
REG_VALUE_DESCR PcmKeyValues[] = {
    ( TEXT( "Dll" )), REG_SZ, ( PBYTE )( TEXT( "ndis.dll" )),
    ( TEXT( "Prefix" )), REG_SZ, ( PBYTE )( TEXT( "NDS" )),
    ( TEXT( "Miniport" )), REG_SZ, ( PBYTE )( TEXT( "NdisDriver" )),
    NULL, 0, NULL
};

// Values for [HKEY_LOCAL_MACHINE\Comm\NdisDriver]
REG_VALUE_DESCR CommKeyValues[] = {
    ( TEXT( "DisplayName" )), REG_SZ, ( PBYTE )
        ( TEXT( "KS8841 Ethernet Driver" )),
    ( TEXT( "Group" )), REG_SZ, ( PBYTE )( TEXT( "NDIS" )),
    ( TEXT( "ImagePath" )), REG_SZ, ( PBYTE )( TEXT( "ks8841.dll" )),
    NULL, 0, NULL
};

// Values for [HKEY_LOCAL_MACHINE\Comm\NdisDriver\Linkage]
REG_VALUE_DESCR LinkageKeyValues[] = {
    ( TEXT( "Route" )), REG_MULTI_SZ, ( PBYTE )( TEXT( "KS88411" )),
    NULL, 0, NULL
};

// Values for [HKEY_LOCAL_MACHINE\Comm\NdisDriver1\Parms]
REG_VALUE_DESCR ParmKeyValues[] = {
    ( TEXT( "BusNumber" )), REG_DWORD, ( PBYTE ) 0,
    ( TEXT( "BusType" )), REG_DWORD, ( PBYTE ) 8,
    ( TEXT( "InterruptNumber" )), REG_DWORD, ( PBYTE ) 9,
    ( TEXT( "IoBaseAddress" )), REG_DWORD, ( PBYTE ) 0x300,
    ( TEXT( "IoLen" )), REG_DWORD, ( PBYTE ) 0x10,
    ( TEXT( "CableType" )), REG_DWORD, ( PBYTE ) 1,
    ( TEXT( "EarlyTransmit" )), REG_DWORD, ( PBYTE ) 0,
    ( TEXT( "FullDuplex" )), REG_DWORD, ( PBYTE ) 0,
    ( TEXT( "LinkIntegrity" )), REG_DWORD, ( PBYTE ) 1,
    NULL, 0, NULL
};

PREG_VALUE_DESCR Values[] = {
#if 0
    PcmKeyValues,
#endif
    CommKeyValues,
    CommKeyValues,
    ParmKeyValues,
    LinkageKeyValues
};

LPWSTR KeyNames[] = {
#if 0
    ( TEXT( "Drivers\\PCMCIA\\KS8841" )),
#endif
    ( TEXT( "Comm\\KS8841" )),
    ( TEXT( "Comm\\KS88411" )),
    ( TEXT( "Comm\\KS88411\\Parms" )),
    ( TEXT( "Comm\\KS8841\\Linkage" ))
};


//
// Add the specified key and its values to the registry under HKEY_LOCAL_MACHINE
//
// NOTE: This function only supports REG_MULTI_SZ strings with one item.
//
BOOL AddKeyValues (
    LPWSTR           KeyName,
    PREG_VALUE_DESCR Vals )
{
    HKEY             hKey;
    DWORD            Status;
    DWORD            dwDisp;
    DWORD            dwVal;
    PREG_VALUE_DESCR pValue;
    DWORD            ValLen;
    PBYTE            pVal;
    LPWSTR           pStr;

    /* create specifed key */
    Status = RegCreateKeyEx( HKEY_LOCAL_MACHINE, KeyName, 0, NULL,
        REG_OPTION_NON_VOLATILE, 0, NULL, &hKey, &dwDisp );

    if ( Status != ERROR_SUCCESS ) {
        return FALSE;
    }

    /* created the key, now put the value in */
    pValue = Vals;
    while ( pValue->val_name ) {
        switch ( pValue->val_type ) {
            case REG_DWORD:
                pVal = ( PBYTE ) &dwVal;
                dwVal = ( DWORD ) pValue->val_data;
                ValLen = sizeof( DWORD );
                break;

            case REG_SZ:
                pVal = ( PBYTE ) pValue->val_data;
                ValLen = ( wcslen(( LPWSTR ) pVal ) + 1 ) * sizeof( WCHAR );
                break;

            case REG_MULTI_SZ:
                dwVal = wcslen(( LPWSTR ) pValue->val_data );
                ValLen = ( dwVal + 2 ) * sizeof( WCHAR );
                pVal = ( PBYTE ) LocalAlloc( LPTR, ValLen );
                if ( pVal == NULL ) {
                    goto akv_fail;
                }
                wcscpy(( LPWSTR ) pVal, ( LPWSTR ) pValue->val_data );
                pStr = ( LPWSTR ) pVal + dwVal;
                pStr[ 1 ] = 0;
                break;
        }
        Status = RegSetValueEx( hKey, pValue->val_name, 0, pValue->val_type,
            pVal, ValLen );
        if ( pValue->val_type == REG_MULTI_SZ ) {
            LocalFree( pVal );
        }

akv_fail:
        if ( Status != ERROR_SUCCESS ) {
            RegCloseKey( hKey );
            return FALSE;
        }

        /* get next value */
        pValue++;
    }

    /* close the key */
    RegCloseKey( hKey );
    return TRUE;
}   // AddKeyValues


#ifdef AUTO_DETECT
//
// Find an unused Detect key number and return the registry path name for it.
//
// Return NULL for failure
//
LPWSTR FindDetectKey ( VOID )
{
    HKEY   hDetectKey;
    DWORD  dwDisp;
    DWORD  dwKeyNum;
    DWORD  Status;
    WCHAR* pKeyName;
    LPWSTR pKeyNum;

    pKeyName = ( WCHAR* ) LocalAlloc( LPTR, sizeof( WCHAR ) * 255 );
    if ( pKeyName == NULL ) {
        return NULL;
    }

    wcscpy( pKeyName, ( TEXT( "Drivers\\PCMCIA\\Detect\\" )));
    pKeyNum = pKeyName + wcslen( pKeyName );

    //
    // Find a detect number and create the detect key.
    //
    for ( dwKeyNum = MIN_DETECT_KEY; dwKeyNum < MAX_DETECT_KEY; dwKeyNum++ ) {
        wsprintf( pKeyNum, ( TEXT( "%02d" )), dwKeyNum );
        Status = RegCreateKeyEx( HKEY_LOCAL_MACHINE, pKeyName, 0, NULL,
            REG_OPTION_NON_VOLATILE, 0, NULL, &hDetectKey, &dwDisp );
        if ( Status == ERROR_SUCCESS ) {
            RegCloseKey( hDetectKey );
            if ( dwDisp == REG_CREATED_NEW_KEY ) {
                return pKeyName;
            }
        }
    }
    LocalFree( pKeyName );
    return NULL;
}  // FindDetectKey


//
// IsInstalled - Search the detect function list in the registry to see if
// we've already been installed.
//
// Return TRUE if this driver's detect function is listed.
//
BOOL IsInstalled ( VOID )
{
    HKEY  hDetectKey;
    HKEY  hDetectNum;
    DWORD dwKeyNum;
    DWORD Status;
    WCHAR KeyNum[ 8 ];
    WCHAR EntryName[ 32 ];
    BOOL  bret;
    DWORD dwLen;
    DWORD dwType;

    Status = RegOpenKeyEx( HKEY_LOCAL_MACHINE,
        TEXT( "Drivers\\PCMCIA\\Detect" ), REG_OPTION_NON_VOLATILE, 0,
        &hDetectKey );
    if ( Status != ERROR_SUCCESS ) {
        DEBUGMSG( ZONE_INIT,
            ( TEXT( "KS884X: IsInstalled RegOpenKeyEx failed %d\r\n" ),
            Status ));
        return FALSE;
    }

    bret = FALSE;
    for ( dwKeyNum = MIN_DETECT_KEY;
            ( bret == FALSE )  &&  ( dwKeyNum < MAX_DETECT_KEY );
            dwKeyNum++ ) {
        wsprintf( KeyNum, ( TEXT( "%02d" )), dwKeyNum );
        Status = RegOpenKeyEx( hDetectKey, KeyNum, REG_OPTION_NON_VOLATILE, 0,
            &hDetectNum );
        if ( Status == ERROR_SUCCESS ) {
            Status = RegQueryValueEx( hDetectNum, TEXT( "Entry" ), 0, &dwType,
                ( PUCHAR ) EntryName, &dwLen );
            if ( Status == ERROR_SUCCESS ) {
                if ( !( wcscmp( EntryName, TEXT( "DetectNdisDevice" ))) ) {
                    bret = TRUE;
                }
            }
            RegCloseKey( hDetectNum );
        }
    }
    RegCloseKey( hDetectKey );
    return bret;
}  // IsInstalled
#endif


//
// Install_Driver function for the NDIS miniport driver.
//
// This function sets up the registry keys and values required to install this
// DLL as a Windows CE plug and play driver.
//
// Input:
//
// LPWSTR lpPnpId - The device's plug and play identifier string.  An install
// function can use lpPnpId to set up a key
// HKEY_LOCAL_MACHINE\Drivers\PCMCIA\<lpPnpId> under the assumption that the
// user will continue to use the same device that generates the same plug and
// play id string.  If there is a general detection method for the card, then
// lpPnpId can be ignored and a detection function can be registered under
// HKEY_LOCAL_MACHINE\Drivers\PCMCIA\Detect.
//
// LPWSTR lpRegPath - Buffer to contain the newly installed driver's device key
// under HKEY_LOCAL_MACHINE in the registry.  Windows CE will attempt to load
// the the newly installed device driver upon completion of its
// Install_Driver function.
//
// DWORD cRegPathSize - Number of bytes in lpRegPath.
//
// Returns lpRegPath if successful, NULL for failure.
//
#ifdef __cplusplus
extern "C"
#endif
LPWSTR Install_Driver (
    LPWSTR lpPnpId,
    LPWSTR lpRegPath,
    DWORD  cRegPathSize )
{
    DWORD  i;

#ifdef AUTO_DETECT
    LPWSTR lpDetectKeyName;
#endif

    DEBUGMSG( ZONE_INIT, ( TEXT( "KS884X: Install_Driver(%s, %s, %d)\r\n" ),
        lpPnpId, lpRegPath, cRegPathSize ));

#ifdef AUTO_DETECT
    if ( IsInstalled() ) {
        return NULL;
    }

    if ( ( lpDetectKeyName = FindDetectKey()) == NULL ) {
        return NULL;
    }

    if ( !AddKeyValues( lpDetectKeyName, DetectKeyValues ) ) {
        goto wid_fail_detect;
    }
#endif

    for ( i = 0; i < ( sizeof( KeyNames ) / sizeof( LPWSTR )); i++ ) {
        if ( !AddKeyValues( KeyNames[ i ], Values[ i ]) ) {
            goto wid_fail;
        }
    }

#ifdef AUTO_DETECT
    LocalFree( lpDetectKeyName );
#endif

    //
    // Return "Drivers\\PCMCIA\\NdisDriver"
    //
    wcscpy( lpRegPath, KeyNames[ 0 ]);
    return lpRegPath;

wid_fail:
    //
    // Clean up after ourself on failure.
    //
    for ( i = 0; i < ( sizeof( KeyNames ) / sizeof( LPWSTR )); i++ ) {
        RegDeleteKey( HKEY_LOCAL_MACHINE, KeyNames[ i ]);
    }

#ifdef AUTO_DETECT

wid_fail_detect:
    RegDeleteKey( HKEY_LOCAL_MACHINE, lpDetectKeyName );
    LocalFree( lpDetectKeyName );
#endif
    return NULL;
}  // Install_Driver


#ifdef AUTO_DETECT
//
// Windows CE Plug and play detection function for the
// NDIS network adapter
//
#ifdef __cplusplus
extern "C"
#endif
LPWSTR DetectNdisDevice (
    CARD_SOCKET_HANDLE hSock,
    UCHAR              DevType,
    LPWSTR             DevKey,
    DWORD              DevKeyLen )
{
    HANDLE hPcmciaDll;
    DWORD cerr;
    GETFIRSTTUPLE pfnGetFirstTuple;
    GETTUPLEDATA pfnGetTupleData;
    UCHAR buf[sizeof(CARD_DATA_PARMS) + 256];
    PCARD_DATA_PARMS pParms;
    PCARD_TUPLE_PARMS pTuple;
    PUCHAR pData;
    LPWSTR ret;

    if (DevType != PCCARD_TYPE_NETWORK) {
        DEBUGMSG(ZONE_INIT,
            (TEXT("DetectXircomCE2 - DevType == %d, expecting %d (PCCARD_TYPE_NETWORK\r\n"),
            DevType, PCCARD_TYPE_NETWORK));
        return NULL;
    }

    hPcmciaDll = LoadLibrary(TEXT("PCMCIA.DLL"));
    if (hPcmciaDll == NULL) {
        DEBUGMSG(ZONE_INIT,
            (TEXT("DetectXircomCE2 - LoadLibrary(PCMCIA.DLL) failed %d\r\n"),
            GetLastError()));
        return NULL;
    }

    pfnGetFirstTuple = (GETFIRSTTUPLE) GetProcAddress(hPcmciaDll, TEXT("CardGetFirstTuple"));
    pfnGetTupleData = (GETTUPLEDATA) GetProcAddress(hPcmciaDll, TEXT("CardGetTupleData"));
    if ((pfnGetFirstTuple == NULL) || (pfnGetTupleData == NULL)) {
        DEBUGMSG(ZONE_INIT,
            (TEXT("DetectXircomCE2 - GetProcAddress() failed %d\r\n")));
        FreeLibrary(hPcmciaDll);
        return NULL;
    }

    ret = NULL;
    pTuple = (PCARD_TUPLE_PARMS)buf;
    pTuple->hSocket = hSock;
    pTuple->uDesiredTuple = CISTPL_VERS_1;
    pTuple->fAttributes = 0;        // Don't want link info.
    cerr = pfnGetFirstTuple(pTuple);
    if (cerr == CERR_SUCCESS) {
        pParms = (PCARD_DATA_PARMS)buf;
        pParms->uBufLen = sizeof(buf) - sizeof(CARD_DATA_PARMS);
        pParms->uTupleOffset = 0;
        cerr = pfnGetTupleData(pParms);
        if (cerr == CERR_SUCCESS) {
            //
            // Check manufacturer string
            //
            pData = buf + sizeof(CARD_DATA_PARMS) + 2;
            if (strstr(pData, "Xircom")) {
                //
                // Check product string
                //
                pData += strlen(pData) + 1;
                if (strstr(pData, "CreditCard")) {
                    //
                    // Check product info string
                    //
                    pData += strlen(pData) + 1;
                    if (strstr(pData, "CE2")) {
                         wcscpy(DevKey, (TEXT("XircomCE2")));
                         Install_Driver(DevKey, (LPWSTR)buf,
                                        sizeof(buf)/sizeof(WCHAR));
                         ret = DevKey;
DEBUGMSG(ZONE_INIT,
    (TEXT("DetectXircomCE2 - Detected a Xircom CreditCard CE2 card.\r\n")));
                    }
                }
            }
        }
    }
    FreeLibrary(hPcmciaDll);
    return ret;
}  // DetectNdisDevice
#endif
#endif

/* ------------------------------------------------------------------------- */

#ifdef __cplusplus
#if defined NDIS51_MINIPORT
#   define DriverStruct3    DriverStruct.Ndis50Chars.Ndis40Chars.Ndis30Chars
#   define DriverStruct4    DriverStruct.Ndis50Chars.Ndis40Chars
#   define DriverStruct5    DriverStruct.Ndis50Chars
#elif defined NDIS50_MINIPORT
#   define DriverStruct3    DriverStruct.Ndis40Chars.Ndis30Chars
#   define DriverStruct4    DriverStruct.Ndis40Chars
#elif defined NDIS40_MINIPORT
#   define DriverStruct3    DriverStruct.Ndis30Chars
#   define DriverStruct4    DriverStruct
#else
#   define DriverStruct3    DriverStruct
#endif

#else
#define DriverStruct3   DriverStruct
#define DriverStruct4   DriverStruct
#define DriverStruct5   DriverStruct
#endif


/*
    This flag indicates whether or not we need to call
    NdisMRegisterAdapterShutdownHandler() and
    NdisMDeregisterAdapterShutdownHandler() to register/deregister the shutdown
    handler.  NDIS 5.1 registers the shutdown handler through the
    NDIS_MINIPORT_CHARACTERISTICS structure.  This will be set to FALSE if we
    successfully register as an NDIS 5.1 miniport.  For NDIS versions < 5.1 (or
    if we need to fall back to NDIS 5.0) this flag will remain TRUE and we will
    explicitly register the shutdown handler.
*/
BOOLEAN ExplicitRegisterShutdownHandler = TRUE;


extern int receive_overrun;


static
VOID PeriodTimerDpc (
    IN  PVOID       SystemSpecific1,
    IN  NDIS_HANDLE hAdapterContext,
    IN  PVOID       SystemSpecific2,
    IN  PVOID       SystemSpecific3 )
{
    // get the pointer to the adapter from the context handle
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hAdapterContext;
    PHARDWARE     pHardware = &pAdapter->m_Hardware;
    NDIS_STATUS   nsStatus;

    if ( !AcquireAdapter( pAdapter, FALSE ) )
    {
        return;
    }

/* PHY change interrupt is working for KS8841. */
#ifdef DEF_KS8842
    if ( !pAdapter->m_Hardware.m_bLinkIntWorking )
        SwitchGetLinkStatus( &pAdapter->m_Hardware );
#endif
    if ( PortReadCounters( &pAdapter->m_Hardware, MAIN_PORT ) )
        goto monitor_release;

#ifdef DEF_KS8842
    if ( PortReadCounters( &pAdapter->m_Hardware, OTHER_PORT ) )
        goto monitor_release;
    if ( PortReadCounters( &pAdapter->m_Hardware, HOST_PORT ) )
        goto monitor_release;
#endif

    if ( pAdapter->m_nPME_wait )
    {
        if ( ++pAdapter->m_nPME_wait >= 2 ) {
            HardwareClearWolPMEStatus( pHardware );

#ifdef DBG
            DbgPrint( "clear PME."NEWLINE );
#endif
            pAdapter->m_nPME_wait = 0;

#ifdef DEBUG_OVERRUN
            DbgPrint( "%u + %u = %u"NEWLINE, pHardware->m_ulReceived,
                pHardware->m_ulDropped,
                pHardware->m_ulReceived + pHardware->m_ulDropped );
            pHardware->m_ulReceived = pHardware->m_ulDropped = 0;
#endif
        }
    }
    else if ( HardwareCheckWolPMEStatus( pHardware ) )
    {
        pAdapter->m_nPME_wait = 1;

#ifdef DBG
        DbgPrint( "PME asserted."NEWLINE );
#endif
    }

monitor_release:
    ReleaseAdapter( pAdapter );

#ifdef DEBUG_OVERRUN
    if ( receive_overrun )
    {
        if ( ++receive_overrun > 2 )
        {
            DbgPrint( "%u + %u = %u"NEWLINE, pHardware->m_ulReceived,
                pHardware->m_ulDropped,
                pHardware->m_ulReceived + pHardware->m_ulDropped );
            pHardware->m_ulReceived = pHardware->m_ulDropped = 0;
            receive_overrun = 0;
        }
    }
#endif
    if ( pAdapter->m_ulNdisMediaState !=
            pHardware->m_ulHardwareState )
    {
        nsStatus = ( NdisMediaStateConnected ==
            pHardware->m_ulHardwareState ) ?
            NDIS_STATUS_MEDIA_CONNECT : NDIS_STATUS_MEDIA_DISCONNECT;

        NdisMIndicateStatus( pAdapter->m_hAdapter, nsStatus, NULL, 0 );
        NdisMIndicateStatusComplete( pAdapter->m_hAdapter );
        pAdapter->m_ulNdisMediaState = pHardware->m_ulHardwareState;
    }
}  // PeriodTimerDpc


static
VOID ResetCompleteTimerDpc (
    IN  PVOID       SystemSpecific1,
    IN  NDIS_HANDLE hAdapterContext,
    IN  PVOID       SystemSpecific2,
    IN  PVOID       SystemSpecific3 )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hAdapterContext;
    PHARDWARE     pHardware = &pAdapter->m_Hardware;

    if ( ( pAdapter->m_ulDriverState & DRIVER_STATE_RESET ) )
    {
        pAdapter->m_ulDriverState &= ~DRIVER_STATE_RESET;
        NdisMResetComplete( pAdapter->m_hAdapter, NDIS_STATUS_SUCCESS, FALSE );
    }
    if ( ( pAdapter->m_ulDriverState & DRIVER_STATE_INIT ) )
    {
        pAdapter->m_ulDriverState &= ~DRIVER_STATE_INIT;
        NdisMSetPeriodicTimer( &pAdapter->m_PeriodTimer, 500 );
    }
}  // ResetCompleteTimerDpc

/* -------------------------------------------------------------------------- */

static struct hw_fn* ks8842_fn = NULL;

/*
    DriverSleep

    Description:
        This routine halts the adapter when the driver is put to sleep.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

VOID DriverSleep (
    IN  PNDIS_ADAPTER pAdapter )
{
    BOOLEAN TimerCanceled;

    // cancel the period timer
    NdisMCancelTimer( &pAdapter->m_PeriodTimer, &TimerCanceled );
    NdisMCancelTimer( &pAdapter->m_ResetTimer, &TimerCanceled );

    ShutdownAdapter( pAdapter );
}  // DriverSleep


/*
    DriverWake

    Description:
        This routine starts the adapter when the driver is woke up.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

VOID DriverWake (
    IN  PNDIS_ADAPTER pAdapter,
    IN  NDIS_DEVICE_POWER_STATE LastPowerState )
{
    if ( NdisDeviceStateD3 == LastPowerState )
        AdapterReset( pAdapter );
    else
        StartAdapter( pAdapter );

    // initialize reset timer.
    NdisMInitializeTimer( &pAdapter->m_ResetTimer, pAdapter->m_hAdapter,
        ResetCompleteTimerDpc, pAdapter );

    // initialize period timer
    NdisMInitializeTimer( &pAdapter->m_PeriodTimer, pAdapter->m_hAdapter,
        PeriodTimerDpc, pAdapter );

    pAdapter->m_ulDriverState |= DRIVER_STATE_INIT;
    NdisMSetTimer( &pAdapter->m_ResetTimer, 1000 );
}  // DriverWake


/*
    MiniportHalt

    Description:
        This routine halts the adapter when the driver is stopped.

    Parameters:
        NDIS_HANDLE hContext
            Handle to adapter context containing adapter information.

    Return (None):
*/

static
VOID MiniportHalt (
    IN  NDIS_HANDLE hContext )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hContext;

#ifdef DEBUG_MINIPORT_HALT
    DbgPrint( "MiniportHalt"NEWLINE );
#endif

    DriverSleep( pAdapter );
    HardwareSetPowerSaving( &pAdapter->m_Hardware, TRUE );

    // Disconnect the interrupt line
    NdisMDeregisterInterrupt( &pAdapter->m_Interrupt );

    // Pause, waiting for any DPC stuff to clear.
    NdisMSleep( 250000 );

    DeRegisterAdapter( pAdapter );

    if ( ExplicitRegisterShutdownHandler )
    {
        NdisMDeregisterAdapterShutdownHandler( pAdapter->m_hAdapter );
    }

#if UNDER_CE
    NdisTerminateWrapper( pAdapter->m_hAdapter, NULL );
#endif
    NdisFreeSpinLock( &pAdapter->m_lockAdapter );
    NdisFreeSpinLock( &pAdapter->m_lockReceive );
    NdisFreeSpinLock( &pAdapter->m_lockHardware );
    FreeMemory( pAdapter );
    NdisFreeMemory( pAdapter, sizeof( NDIS_ADAPTER ), 0 );
}  // MiniportHalt


/*
    MiniportShutdown

    Description:
        This routine stops the adapter when the driver is stopped.

        Notes: MiniportShutdown SHOULD CALL NO NDISXXX FUNCTIONS.

    Parameters:
        NDIS_HANDLE hContext
            Handle to adapter context containing adapter information.

    Return (None):
*/

static
VOID MiniportShutdown (
    IN  NDIS_HANDLE hContext )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hContext;

#if DBG
    DbgPrint( "MiniportShutdown"NEWLINE );
#endif

    ShutdownAdapter( pAdapter );
    HardwareSetPowerSaving( &pAdapter->m_Hardware, TRUE );
}  // MiniportShutdown


#if (NDISVER >= 51)
/*
    MiniportPnPEventNotify

    Description:
        This routine notifies the device about PnP events.

    Parameters:
        NDIS_HANDLE hContext
            Handle to adapter context containing adapter information.

        NDIS_DEVICE_PNP_EVENT PnPEvent

        PVOID pBuffer
            Pointer to a ULONG variable.

        ULONG ulBufferLength
            The number of bytes in pBuffer.

    Return (None):
*/

static
VOID MiniportPnPEventNotify (
    IN  NDIS_HANDLE           hContext,
    IN  NDIS_DEVICE_PNP_EVENT PnPEvent,
    IN  PVOID                 pBuffer,
    IN  ULONG                 ulBufferLength )
{
#if 0
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hContext;
#endif

    if ( NdisDevicePnPEventPowerProfileChanged == PnPEvent )
    {
        ULONG profile = *(( ULONG* ) pBuffer );

        if ( NdisPowerProfileBattery == profile )
        {
#if DBG
            DBG_PRINT( "battery"NEWLINE );
#endif
        }
        else if ( NdisPowerProfileAcOnLine == profile )
        {
#if DBG
            DBG_PRINT( "AC"NEWLINE );
#endif
        }
    }
    else if ( NdisDevicePnPEventSurpriseRemoved == PnPEvent )
    {
#if DBG
        DBG_PRINT( "Removed!"NEWLINE );
#endif
    }
}  // MiniportPnPEventNotify
#endif


/*
    MiniportInitialize

    Description:
        This function initializes the adapter and registers resources with the
        NDIS wrapper.

    Parameters:
        PNDIS_STATUS pnsOpenErrorStatus
            Extra status bytes for opening token ring adapters.

        PUINT puiSelectedMediumIndex
            Index of the media type chosen by the driver.

        PNDIS_MEDIUM pnmMediumArray
            Array of media types for the driver to choose from.

        UINT uiMediumArraySize
            Number of entries in the array.

        NDIS_HANDLE hAdapter
            Handle to adapter to pass to the NDIS wrapper.

        NDIS_HANDLE hConfiguration
            Handle of configuration to pass to NdisOpenConfiguration.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS if successful; otherwise an error code indicating
        failure.
*/

static
NDIS_STATUS MiniportInitialize (
    OUT PNDIS_STATUS pnsOpenErrorStatus,
    OUT PUINT        puiSelectedMediumIndex,
    IN  PNDIS_MEDIUM pnmMediumArray,
    IN  UINT         uiMediumArraySize,
    IN  NDIS_HANDLE  hAdapter,
    IN  NDIS_HANDLE  hConfiguration )
{
    PNDIS_ADAPTER pAdapter;
    NDIS_STATUS   nsStatus;
    UINT          i;
    UCHAR         bInterrupt = FALSE;

#if (NDISVER < 50)
    NDIS_PHYSICAL_ADDRESS HighestAcceptableMax =
        NDIS_PHYSICAL_ADDRESS_CONST( -1, -1 );
#endif

    for ( i = 0; i < uiMediumArraySize; i++ )
    {
        if ( pnmMediumArray[ i ] == NdisMedium802_3 )
            break;
    }
    if ( i == uiMediumArraySize )
    {
        return NDIS_STATUS_UNSUPPORTED_MEDIA;
    }
    *puiSelectedMediumIndex = i;

    // Allocate memory for the adapter block now.

#if (NDISVER >= 50)
    nsStatus = NdisAllocateMemoryWithTag(( PVOID* ) &pAdapter,
        sizeof( NDIS_ADAPTER ), 'rciM' );
#else
    nsStatus = NdisAllocateMemory(( PVOID* ) &pAdapter, sizeof( NDIS_ADAPTER ),
        0, HighestAcceptableMax );
#endif
    if ( nsStatus != NDIS_STATUS_SUCCESS )
    {
        return( nsStatus );
    }

    // Clear out the adapter block, which sets all default values to FALSE
    // or NULL.

    NdisZeroMemory( pAdapter, sizeof( NDIS_ADAPTER ));

    pAdapter->m_Hardware.m_hwfn = ks8842_fn;

    // Allocate the spin locks.
    NdisAllocateSpinLock( &pAdapter->m_lockAdapter );
    NdisAllocateSpinLock( &pAdapter->m_lockReceive );
    NdisAllocateSpinLock( &pAdapter->m_lockHardware );

    // Process the configuration parameters.
    nsStatus = InitAdapter( pAdapter, hAdapter, hConfiguration );
    if ( nsStatus != NDIS_STATUS_SUCCESS )
    {
        goto MiniportInitializeMemoryError;
    }

    // Register the adapter.
    nsStatus = RegisterAdapter( pAdapter );
    if ( nsStatus != NDIS_STATUS_SUCCESS )
    {
        goto MiniportInitializeError;
    }

    nsStatus = NdisMRegisterInterrupt( &pAdapter->m_Interrupt,
        pAdapter->m_hAdapter, pAdapter->m_ulInterruptNumber,
        pAdapter->m_ulInterruptNumber, TRUE, TRUE,
        NdisInterruptLevelSensitive );
    if ( nsStatus != NDIS_STATUS_SUCCESS )
    {
#if 0
        NdisWriteErrorLogEntry( pAdapter->m_hAdapter,
            NDIS_ERROR_TRILOG_NO_INTERRUPT_ROUTE, 3, 0x1056000C, TRUE,
            pAdapter->m_ulInterruptNumber );
#endif
        goto MiniportInitializeError;
    }
    bInterrupt = TRUE;

    nsStatus = SetupAdapter( pAdapter );
    if ( nsStatus != NDIS_STATUS_SUCCESS )
    {
        goto MiniportInitializeError;
    }

    DriverWake( pAdapter, NdisDeviceStateD0 );

    // Register shutdown handler.  Note: NDIS 5.1 registers the handler in
    // NDIS_MINIPORT_CHARACTERISTICS
    if ( ExplicitRegisterShutdownHandler )
    {
        NdisMRegisterAdapterShutdownHandler( hAdapter, pAdapter,
            MiniportShutdown );
    }

    return NDIS_STATUS_SUCCESS;

MiniportInitializeError:
    if ( bInterrupt )
        NdisMDeregisterInterrupt( &pAdapter->m_Interrupt );
    DeRegisterAdapter( pAdapter );

#if UNDER_CE
    NdisTerminateWrapper( pAdapter->m_hAdapter, NULL );
#endif

MiniportInitializeMemoryError:
    FreeMemory( pAdapter );

    NdisFreeSpinLock( &pAdapter->m_lockAdapter );
    NdisFreeSpinLock( &pAdapter->m_lockReceive );
    NdisFreeSpinLock( &pAdapter->m_lockHardware );
    NdisFreeMemory( pAdapter, sizeof( NDIS_ADAPTER ), 0 );
    return( nsStatus );
}  // MiniportInitialize

/* -------------------------------------------------------------------------- */

static
VOID DriverUnload (
    IN  PDRIVER_OBJECT DriverObject )
{
#if DBG
    DBG_PRINT( "DriverUnload"NEWLINE );
#endif
    if ( ks8842_fn )
    {
        NdisFreeMemory( ks8842_fn, sizeof( struct hw_fn ), 0 );
        ks8842_fn = NULL;
    }
}  /* DriverUnload */


/*
    DriverEntry

    Description:
        This function is the main entry point of the driver.

    Parameters:
        PDRIVER_OBJECT pDriverObject
            Pointer to driver object.

        PUNICODE_STRING RegistryPath
            Pointer to unicode string containing the register path that stores
            the configuration of the driver.

    Return (NTSTATUS):
        STATUS_SUCCESS if successful; otherwise STATUS_UNSUCCESSFUL.
*/

#ifdef __cplusplus
extern "C"
#endif
NTSTATUS DriverEntry (
    IN  PDRIVER_OBJECT  DriverObject,
    IN  PUNICODE_STRING RegistryPath )
{
    // Characteristics table for this driver.
    NDIS_MINIPORT_CHARACTERISTICS DriverStruct;
    NDIS_HANDLE                   NdisWrapperHandle;
    NDIS_STATUS                   nsStatus;

#ifdef DBG
    DBG_PRINT( "%s"NEWLINE, version );
#endif

    // Initialize the wrapper.
    NdisMInitializeWrapper( &NdisWrapperHandle, DriverObject, RegistryPath,
        NULL );

    // Clear the characteristics table to ensure there's no bogus data passed
    // to NDIS.
    NdisZeroMemory( &DriverStruct, sizeof( DriverStruct ));

    // Initialize the Miniport characteristics for the call to
    // NdisMRegisterMiniport.
    DriverStruct3.MajorNdisVersion          = NDIS_MAJOR_VERSION;
    DriverStruct3.MinorNdisVersion          = NDIS_MINOR_VERSION;
    DriverStruct3.CheckForHangHandler       = MiniportCheckForHang;
    DriverStruct3.DisableInterruptHandler   = MiniportDisableInterrupt;
    DriverStruct3.EnableInterruptHandler    = MiniportEnableInterrupt;
    DriverStruct3.HaltHandler               = MiniportHalt;
    DriverStruct3.HandleInterruptHandler    = MiniportHandleInterrupt;
    DriverStruct3.InitializeHandler         = MiniportInitialize;
    DriverStruct3.ISRHandler                = MiniportISR;
    DriverStruct3.QueryInformationHandler   = MiniportQueryInformation;
    DriverStruct3.ReconfigureHandler        = NULL;
    DriverStruct3.ResetHandler              = MiniportReset;
    DriverStruct3.SendHandler               = MiniportSend;
    DriverStruct3.SetInformationHandler     = MiniportSetInformation;
    DriverStruct3.TransferDataHandler       = NULL;

    // NDIS 4.0 extensions...
    //
    // If you want to support NDIS 3.1 then don't support these functions and
    // change the NDIS_MAJOR_VERSION & NDIS_MINOR_VERSION and change the compile
    // and link parameters to reflect 3.1.  NDIS 3.1 uses SendHandler to send
    // packets (and its implemented).
    //
    // NDIS 4.0 does give a speed boost because the SendPacketsHandler will
    // handle multiple packets at a time.

#if (NDISVER >= 40)
    // Specific to NDIS 4
    DriverStruct4.ReturnPacketHandler       = MiniportReturnPacket;
    DriverStruct4.SendPacketsHandler        = MiniportSendPackets;
    DriverStruct4.AllocateCompleteHandler   = NULL;
#endif

#if (NDISVER >= 51)
    // Specific to NDIS 5.1
    DriverStruct.AdapterShutdownHandler     = MiniportShutdown;
    DriverStruct.CancelSendPacketsHandler   = NULL;
    DriverStruct.PnPEventNotifyHandler      = MiniportPnPEventNotify;
#endif

    nsStatus = NdisMRegisterMiniport( NdisWrapperHandle, &DriverStruct,
        sizeof( DriverStruct ));

#if (NDISVER >= 51)
    // Fall back to NDIS 5.0 support if necessary.
    if ( nsStatus != NDIS_STATUS_SUCCESS )
    {

#if DBG
        DbgPrint( "DriverEntry falling back to NDIS 5.0..."NEWLINE );
#endif
        DriverStruct3.MajorNdisVersion = 5;
        DriverStruct3.MinorNdisVersion = 0;
        nsStatus = NdisMRegisterMiniport( NdisWrapperHandle, &DriverStruct,
            sizeof( DriverStruct5 ));
    }
    else
    {
        // Don't need to explicitly register shutdown handler -- already done in
        // NDIS_MINIPORT_CHARACTERISTICS.
        ExplicitRegisterShutdownHandler = FALSE;
    }
#endif

    if ( NDIS_STATUS_SUCCESS == nsStatus )
    {

#if (NDISVER >= 50)
        NdisAllocateMemoryWithTag(( PVOID* ) &ks8842_fn,
            sizeof( struct hw_fn ), 'rciM' );
#else
        NDIS_PHYSICAL_ADDRESS HighestAcceptableMax =
            NDIS_PHYSICAL_ADDRESS_CONST( -1, -1 );

        NdisAllocateMemory(( PVOID* ) &ks8842_fn,
            sizeof( struct hw_fn ),
            0, HighestAcceptableMax );
#endif
        HardwareSetupFunc( ks8842_fn );

        NdisMRegisterUnloadHandler( NdisWrapperHandle, DriverUnload );
    }

#if DBG
#ifdef UNDER_CE
    DEBUGMSG( ZONE_INIT, ( TEXT( "KS884X: DriverEntry:%x\r\n" ), nsStatus ));

#else
    DbgPrint( "%p = ", DriverObject );
    switch ( nsStatus )
    {
        case NDIS_STATUS_BAD_CHARACTERISTICS:
            DbgPrint( "NDIS_STATUS_BAD_CHARACTERISTICS\n" );
            break;
        case NDIS_STATUS_BAD_VERSION:
            DbgPrint( "NDIS_STATUS_BAD_VERSION\n" );
            break;
        case NDIS_STATUS_RESOURCES:
            DbgPrint( "NDIS_STATUS_RESOURCES\n" );
            break;
        case NDIS_STATUS_FAILURE:
            DbgPrint( "NDIS_STATUS_FAILURE\n" );
            break;
        case NDIS_STATUS_SUCCESS:
            DbgPrint( "NDIS_STATUS_SUCCESS\n" );
            break;
        default:
            DbgPrint( "NDIS Unknown:%08x\n", nsStatus );
            break;
    }
#endif
#endif

    return( nsStatus == NDIS_STATUS_SUCCESS ? STATUS_SUCCESS :
        STATUS_UNSUCCESSFUL );
}  // DriverEntry
