/* ---------------------------------------------------------------------------
          Copyright (c) 2003-2007 Micrel, Inc.  All rights reserved.
   ---------------------------------------------------------------------------

    NdisSend.c - NDIS driver send functions.

    Author      Date        Description
    THa         12/01/03    Created file.
    THa         06/27/05    Updated for version 0.1.5.
    THa         02/23/06    Updated for WinCE 5.0.
    THa         06/02/06    Report statistics from MIB counters.
    THa         12/18/07    Update for 64-bit systems.
   ---------------------------------------------------------------------------
*/


#include "target.h"
#include "NdisDevice.h"


/* -------------------------------------------------------------------------- */

USHORT ntohs (
    USHORT wNetValue )
{
    USHORT wHostValue;

    wHostValue = (( UCHAR ) wNetValue << 8 ) | ( UCHAR )( wNetValue >> 8 );
    return( wHostValue );
}  // ntohs


/*
    GetFirstBuffer

    Description:
        This function retrieves the first NDIS buffer in the NDIS packet.

    Parameters:
        PNDIS_BUFFER* ppndisBuffer
            Pointer to NDIS buffer pointer.

        PUINT puiLength
            Buffer to store the length of the data buffer.

    Return (PUCHAR):
        Pointer to data buffer in the NDIS buffer.
*/

__inline
PUCHAR GetFirstBuffer (
    PNDIS_BUFFER* ppndisBuffer,
    PUINT         puiLength )
{
    PUCHAR pBuffer;

    // Each packet may be made up of several buffers, linked together.  Get the
    // virtual address (in pBuffer) and the length (in uiLength) of the first
    // buffer in the packet by calling NdisQueryBuffer.
    NdisQueryBuffer( *ppndisBuffer, ( PVOID* ) &pBuffer, puiLength );

    // If the pointer to the buffer is not NULL and the length returned from
    // the above call is zero, skip to the next buffer and query it for its
    // length.
    while ( !*puiLength )
    {
        NdisGetNextBuffer( *ppndisBuffer, ppndisBuffer );
        if ( !*ppndisBuffer )
        {
            return NULL;
        }
        NdisQueryBuffer( *ppndisBuffer, ( PVOID* ) &pBuffer, puiLength );
    }
    return( pBuffer );
}  // GetFirstBuffer


/*
    GetNextBuffer

    Description:
        This function retrieves the next NDIS buffer in the NDIS packet.

    Parameters:
        PNDIS_BUFFER* ppndisBuffer
            Pointer to NDIS buffer pointer.

        PUINT puiLength
            Buffer to store the length of the data buffer.

    Return (PUCHAR):
        Pointer to data buffer in the NDIS buffer.
*/

__inline
PUCHAR GetNextBuffer (
    PNDIS_BUFFER* ppndisBuffer,
    PUINT         puiLength )
{
    PUCHAR pBuffer;

    *puiLength = 0;

    // Go through the process of eliminating buffers with no data in them.
    // Exit the following while loop when we run out of buffers or have one
    // that contains some data.
    while ( !*puiLength )
    {
        NdisGetNextBuffer( *ppndisBuffer, ppndisBuffer );
        if ( !*ppndisBuffer )
        {
            return NULL;
        }
        NdisQueryBuffer( *ppndisBuffer, ( PVOID* ) &pBuffer, puiLength );
    }
    return( pBuffer );
}  // GetNextBuffer


/*
    AdapterCopyDownPacket

    Description:
        This function copies the packet down to the adapter buffer.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        PNDIS_BUFFER pndisBuffer
            Pointer to NDIS buffer.

        UINT uiPacketLength
            The length of the buffer.

        PUINT puiLength
            Buffer to store the actual length of data copied.

        BOOLEAN* pbBroadcast
            Buffer to store the broadcast flag.

    Return (BOOLEAN):
        TRUE if successful; otherwise, FALSE.
*/

BOOLEAN AdapterCopyDownPacket (
    IN  PNDIS_ADAPTER pAdapter,
    IN  PNDIS_BUFFER  pndisBuffer,
    IN  UINT          uiPacketLength,
    OUT PUINT         puiLength,
    OUT BOOLEAN*      pbBroadcast )
{
    PHARDWARE pHardware = &pAdapter->m_Hardware;

    // Length of current source buffer
    UINT      uiBufferLength;

    // Address of current source buffer
    PUCHAR    pBuffer;

    *puiLength = 0;

    pBuffer = GetFirstBuffer( &pndisBuffer, &uiBufferLength );

    if ( pBuffer )
    {
        if ( ( pBuffer[ 0 ] & 0x01 ) )
        {
            *pbBroadcast = TRUE;
        }
        else
        {
            *pbBroadcast = FALSE;
        }
        HardwareSetTransmitLength( pHardware, uiPacketLength );
    }

    while ( pBuffer )
    {
        MOVE_MEM( &pHardware->m_bLookahead[ *puiLength ], pBuffer,
            uiBufferLength );
        *puiLength += uiBufferLength;
        pBuffer = GetNextBuffer( &pndisBuffer, &uiBufferLength );
    }

    if ( !*puiLength )
    {
        goto CopyDownPacketError;
    }
    HW_WRITE_BUFFER( pHardware, REG_DATA_OFFSET, pHardware->m_bLookahead,
        *puiLength );

#ifdef SH_16BIT_WRITE
    HW_WRITE_WORD( pHardware, REG_TXQ_CMD_OFFSET, TXQ_CMD_ENQUEUE_PACKET );

#else
    HW_WRITE_BYTE( pHardware, REG_TXQ_CMD_OFFSET, TXQ_CMD_ENQUEUE_PACKET );
#endif

#if defined( DEBUG_TX )  &&  defined( UNDER_CE )
DBG_PRINT( "%04d: %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X"NEWLINE,
    *puiLength,
    pHardware->m_bLookahead[ 0 ], pHardware->m_bLookahead[ 1 ],
    pHardware->m_bLookahead[ 2 ], pHardware->m_bLookahead[ 3 ],
    pHardware->m_bLookahead[ 4 ], pHardware->m_bLookahead[ 5 ],
    pHardware->m_bLookahead[ 6 ], pHardware->m_bLookahead[ 7 ],
    pHardware->m_bLookahead[ 8 ], pHardware->m_bLookahead[ 9 ],
    pHardware->m_bLookahead[ 10 ], pHardware->m_bLookahead[ 11 ],
    pHardware->m_bLookahead[ 12 ], pHardware->m_bLookahead[ 13 ],
    pHardware->m_bLookahead[ 14 ], pHardware->m_bLookahead[ 15 ],
    pHardware->m_bLookahead[ 16 ], pHardware->m_bLookahead[ 17 ]);
#endif

    return TRUE;

CopyDownPacketError:

#ifdef DEBUG_COUNTER
    pHardware->m_nBad[ COUNT_BAD_COPY_DOWN ]++;
#endif
    return FALSE;
}  // AdapterCopyDownPacket

/* -------------------------------------------------------------------------- */

/*
    SendPacket

    Description:
        This function sends a packet to hardware.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        PNDIS_BUFFER pBuffer
            Pointer to NDIS buffer.

        UINT uiPacketLength
            The length of the buffer.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS if successful; otherwise, an error code indicating
        failure.
*/

NDIS_STATUS SendPacket (
    PNDIS_ADAPTER pAdapter,
    PNDIS_BUFFER  pBuffer,
    UINT          uiPacketLength )
{
    PHARDWARE   pHardware = &pAdapter->m_Hardware;
    NDIS_STATUS nsStatus = NDIS_STATUS_FAILURE;
    UINT        uiLength;
    BOOLEAN     bBroadcast;
    BOOLEAN     bResult;
    int         port = MAIN_PORT;

    bResult = AdapterCopyDownPacket( pAdapter, pBuffer, uiPacketLength,
        &uiLength, &bBroadcast );

    if ( !bResult )
    {
        goto SendIntrDone;
    }

#if 0
    if ( HardwareSendPacket( pHardware ) )
#endif
    {
        nsStatus = NDIS_STATUS_SUCCESS;
        if ( uiLength )
        {
            pHardware->m_cnCounter[ port ][ OID_COUNTER_XMIT_OK ]++;

#ifndef NO_STATS
            if ( bBroadcast )
            {
                pHardware->m_cnCounter[ port ]
                    [ OID_COUNTER_BROADCAST_FRAME_XMIT ]++;
                pHardware->m_cnCounter[ port ]
                    [ OID_COUNTER_BROADCAST_BYTES_XMIT ] += uiLength;
            }
            else
            {
                pHardware->m_cnCounter[ port ]
                    [ OID_COUNTER_DIRECTED_FRAMES_XMIT ]++;
                pHardware->m_cnCounter[ port ]
                    [ OID_COUNTER_DIRECTED_BYTES_XMIT ] += uiLength;
            }
#endif
        }
    }

SendIntrDone:

    return( nsStatus );
}  // SendPacket

/* -------------------------------------------------------------------------- */

/*
    MiniportSend

    Description:
        This function sends one packet in NDIS 3.0.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        PNDIS_PACKET pPacket
            Pointer to NDIS packet.

        UINT uiFlags
            Send flags.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS if successful; otherwise, an error code indicating
        failure.
*/

NDIS_STATUS MiniportSend (
    IN  NDIS_HANDLE  hAdapterContext,
    IN  PNDIS_PACKET pPacket,
    IN  UINT         uiFlags )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hAdapterContext;
    PHARDWARE     pHardware = &pAdapter->m_Hardware;
    NDIS_STATUS   nsStatus = NDIS_STATUS_FAILURE;
    PNDIS_BUFFER  pndisBuffer;
    UINT          uiPacketLength;
    UINT          InterruptMask;

    if ( !AcquireAdapter( pAdapter, FALSE ) )
    {
        goto SendDone;
    }

#ifdef DEBUG_COUNTER
    pHardware->m_nGood[ COUNT_GOOD_SEND_PACKET ]++;
#endif
    NdisQueryPacket( pPacket, NULL, NULL, &pndisBuffer, &uiPacketLength );

    // If the returned packet length is zero, there is nothing to transmit.
    if ( !uiPacketLength )
    {
#ifdef DEBUG_COUNTER
        pHardware->m_nGood[ COUNT_GOOD_SEND_ZERO ]++;
#endif
        nsStatus = NDIS_STATUS_SUCCESS;
        goto SendLockDone;
    }

    /* Save the current interrupt mask and block all interrupts. */
    InterruptMask = HardwareBlockInterrupt( pHardware );

    if ( !HardwareAllocPacket( pHardware, uiPacketLength ) )
    {
        nsStatus = NDIS_STATUS_RESOURCES;
        goto SendBlockDone;
    }

    nsStatus = SendPacket( pAdapter, pndisBuffer, uiPacketLength );

SendBlockDone:
    /* Restore the interrupt mask. */
    HardwareSetInterrupt( pHardware, InterruptMask );

SendLockDone:
    ReleaseAdapter( pAdapter );

SendDone:
    NDIS_SET_PACKET_STATUS( pPacket, nsStatus );
    return( nsStatus );
}  // MiniportSend

/* -------------------------------------------------------------------------- */

/*
    AdapterTransmitQueueLength

    Description:
        This function returns the number of packets waiting in the transmit
        queue.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (UINT):
        The number of packets in the send queue.
*/

UINT AdapterTransmitQueueLength (
    IN  PNDIS_ADAPTER pAdapter )
{
    PNDIS_PACKET pPacket;
    UINT         ulCount = 0;

    if ( !AcquireAdapter( pAdapter, FALSE ) )
        return( 4 );

    pPacket = pAdapter->m_pPacketQueueHead;
    while ( pPacket )
    {
        ++ulCount;
        pPacket = RESERVED( pPacket )->Next;
    }
    ReleaseAdapter( pAdapter );
    return( ulCount );
}  // AdapterTransmitQueueLength


#ifdef SEND_QUEUE
/*
    AddPacketToTransmitQueue

    Description:
        This routine adds packets to the transmit queue.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        PNDIS_PACKET pPacket
            Pointer to NDIS packet.

    Return (None):
*/

void AddPacketToTransmitQueue (
    PNDIS_ADAPTER pAdapter,
    PNDIS_PACKET  pPacket )
{
    if ( pAdapter->m_pPacketQueueTail )
        RESERVED( pAdapter->m_pPacketQueueTail )->Next = pPacket;
    else
        pAdapter->m_pPacketQueueHead = pPacket;
    RESERVED( pPacket )->Next = NULL;
    pAdapter->m_pPacketQueueTail = pPacket;
    NDIS_SET_PACKET_STATUS( pPacket, NDIS_STATUS_PENDING );
}  // AddPacketToTransmitQueue


/*
    SendNextPacket

    Description:
        This routine is used to send the next packet waiting in the queue.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

void SendNextPacket (
    PNDIS_ADAPTER pAdapter )
{
    PHARDWARE    pHardware = &pAdapter->m_Hardware;
    NDIS_STATUS  nsStatus;
    PNDIS_PACKET pPacket;
    PNDIS_BUFFER pndisBuffer;
    UINT         uiPacketLength;

    if ( !AcquireAdapter( pAdapter, FALSE ) )
    {
        return;
    }

    // Return if we don't have a packet to send.
    if ( !pAdapter->m_pPacketQueueHead )
    {
#ifdef DEBUG_COUNTER
        pHardware->m_nGood[ COUNT_GOOD_NO_NEXT_PACKET ]++;
#endif
        goto SendNextPacketDone;
    }

    pPacket = pAdapter->m_pPacketQueueHead;

    NdisQueryPacket( pPacket, NULL, NULL, &pndisBuffer, &uiPacketLength );

    // If the returned packet length is zero, there is nothing to transmit.
    if ( !uiPacketLength )
    {
#ifdef DEBUG_COUNTER
        pHardware->m_nGood[ COUNT_GOOD_SEND_ZERO ]++;
#endif
        nsStatus = NDIS_STATUS_SUCCESS;
        NDIS_SET_PACKET_STATUS( pPacket, nsStatus );
        goto SendNextDone;
    }

    if ( !HardwareAllocPacket( pHardware, uiPacketLength ) )
    {
        goto SendNextPacketDone;
    }

    nsStatus = SendPacket( pAdapter, pndisBuffer, uiPacketLength );

SendNextDone:

    // Remove the packet from the queue.
#ifdef DEBUG_COUNTER
    pHardware->m_nGood[ COUNT_GOOD_NEXT_PACKET ]++;
#endif
    pAdapter->m_pPacketQueueHead = RESERVED( pPacket )->Next;
    if ( pPacket == pAdapter->m_pPacketQueueTail )
        pAdapter->m_pPacketQueueTail = NULL;

    NdisMSendComplete( pAdapter->m_hAdapter, pPacket, nsStatus );

SendNextPacketDone:
    ReleaseAdapter( pAdapter );
}  // SendNextPacket
#endif


/*
    MiniportSendPackets

    Description:
        This routine sends one or more packets in NDIS 4.0.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        PPNDIS_PACKET pPacketArray
            Array of pointers to NDIS packets.

        UINT uiCount
            Number of packets in the array.

    Return (None):
*/

VOID MiniportSendPackets (
    IN  NDIS_HANDLE   hAdapterContext,
    IN  PPNDIS_PACKET pPacketArray,
    IN  UINT          uiCount )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hAdapterContext;
    NDIS_STATUS   nsStatus;
    PNDIS_PACKET  pPacket;
    PNDIS_BUFFER  pndisBuffer;
    PHARDWARE     pHardware = &pAdapter->m_Hardware;
    UINT          uiIndex;
    UINT          uiPacketLength;
    UINT          InterruptMask;

//DbgPrint( "MiniportSendPackets\n" );

    // If the count of packets in the array is zero, exit after noting this
    // fact.
    if ( !uiCount )
    {

#ifdef DEBUG_COUNTER
        pHardware->m_nBad[ COUNT_BAD_SEND_ZERO ]++;
#endif
        return;
    }

    // Acquire the hardware to avoid having 2 threads messing with the adapter
    // at the same time.  If this can't happen, return all of the packets with
    // a failure indication and exit.
    if ( !AcquireAdapter( pAdapter, FALSE ) )
    {
        for ( uiIndex = 0; uiIndex < uiCount; uiIndex++ )
        {
            pPacket = pPacketArray[ uiIndex ];
            NDIS_SET_PACKET_STATUS( pPacket, NDIS_STATUS_FAILURE );
        }
        return;
    }

    /* Save the current interrupt mask and block all interrupts. */
    InterruptMask = HardwareBlockInterrupt( pHardware );

    // The following 'for' loop handles one packet in the array each time
    // through.

    for ( uiIndex = 0; uiIndex < uiCount; uiIndex++ )
    {
        // Count packets sent.  Point pPacket to the next packet in the array.
#ifdef DEBUG_COUNTER
        pHardware->m_nGood[ COUNT_GOOD_SEND_PACKET ]++;
#endif
        pPacket = pPacketArray[ uiIndex ];

#ifdef SEND_QUEUE
        // If there are packets waiting in the queue to be transmitted add this
        // packet to the end of that queue.  We have to keep things in order.
        if ( pAdapter->m_pPacketQueueTail )
        {
#ifdef DEBUG_COUNTER
            pHardware->m_nGood[ COUNT_GOOD_SEND_QUEUE ]++;
#endif
            AddPacketToTransmitQueue( pAdapter, pPacket );
        }
        else
#endif
        {
            NdisQueryPacket( pPacket, NULL, NULL, &pndisBuffer,
                &uiPacketLength );

            // If the returned packet length is zero, there is nothing to
            // transmit.
            if ( !uiPacketLength )
            {
#ifdef DEBUG_COUNTER
                pHardware->m_nGood[ COUNT_GOOD_SEND_ZERO ]++;
#endif
                NDIS_SET_PACKET_STATUS( pPacket, NDIS_STATUS_SUCCESS );
                continue;
            }

            if ( !HardwareAllocPacket( pHardware, uiPacketLength ) )
            {

#ifdef SEND_QUEUE
                AddPacketToTransmitQueue( pAdapter, pPacket );

#else
                NDIS_SET_PACKET_STATUS( pPacket, NDIS_STATUS_FAILURE );
#endif
            }
            else
            {
                nsStatus = SendPacket( pAdapter, pndisBuffer, uiPacketLength );

                NDIS_SET_PACKET_STATUS( pPacket, nsStatus );
            }
        }
    }

    /* Restore the interrupt mask. */
    HardwareSetInterrupt( pHardware, InterruptMask );

    ReleaseAdapter( pAdapter );
}  // MiniportSendPackets

