/* Copyright (C) 2010 - 2011 Broadcom Corporation.
 * Portions Copyright (C) VMware, Inc. 2007-2011. All Rights Reserved.
 */

#define TG3_NETQ_WAIT_EVENT_TIMEOUT	msecs_to_jiffies(100)
#define TG3_VRQ_CHANGE_TIMEOUT_US	10000
#define TG3_VRQ_FLUSH_TIMEOUT_US	50000
#if 0
#define TG3_VRQ_MAX_NUM_TX_QS(tp)	(min_t(u16, (tp)->irq_cnt - 1, 15))
#else
#define TG3_VRQ_MAX_NUM_TX_QS(tp)	0
#endif
#define TG3_VRQ_MAX_NUM_RX_QS(tp)	(tp->irq_cnt - 1)

static void tg3_vmware_timer(struct tg3 *tp)
{
	/*
	 * If the driver says this NIC is in promiscuous mode, the hardware
	 * better agree with it. See PR 88299 for more details, but there
	 * are cases (eg. bcm5704 rev10) that the set promiscuous bit is
	 * lost for no reason and this seems to be the least intrusive
	 * workaround for that issue (likely a hardware bug).
	 *
	 * This is happening on other cards as well (BCM5700 rev04) PR 97750
	 * Also, on some cards (5703) we have seen other bits getting messed up
	 */
	if (netif_carrier_ok(tp->dev)) {
		u32 rx_mode = tr32(MAC_RX_MODE);
		if (!(rx_mode & RX_MODE_PROMISC) && (rx_mode != tp->rx_mode)) {
			/*
			 * We love to warn the users every time there is such a
			 * register reset, but we do not want to do it forever
			 * given some NICs are really that bad (PR 105488).
			 */
			if (tp->vmware.rx_mode_reset_counter < 200) {
				netdev_info(tp->dev, "%s: rx_mode "
					    "0x%x(%s)=>0x%x tg3_flags 0x%x "
					    "tg3_flags2 0x%x\n",
					    tp->dev->name, rx_mode,
				       rx_mode & RX_MODE_PROMISC ? "on" : "off",
					    tp->rx_mode, tp->tg3_flags,
					    tp->tg3_flags2);
				tp->vmware.rx_mode_reset_counter++;
			}
			tw32_f(MAC_RX_MODE, tp->rx_mode);
		}
	}

	/*
	 * we've seen ssome 5703's (rev 10) which don't seen to
	 * generate an interrupt on link-up state changes. bug 89197.
	 */
	if (!netif_carrier_ok(tp->dev) &&
	    !(tp->tg3_flags &
	     (TG3_FLAG_USE_LINKCHG_REG | TG3_FLAG_POLL_SERDES))) {
		struct tg3_hw_status *sblk = tp->napi[0].hw_status;
		if (sblk->status & SD_STATUS_LINK_CHG) {
			sblk->status = SD_STATUS_UPDATED |
			       (sblk->status & ~SD_STATUS_LINK_CHG);
			tg3_setup_phy(tp, 0);
		}
	}
}

#if !defined(TG3_VMWARE_BMAPILNX_DISABLE)
static int
tg3_vmware_ioctl_cim(struct net_device *dev, struct ifreq *ifr)
{
	struct tg3 *tp = netdev_priv(dev);
	void __user *useraddr = ifr->ifr_data;
	struct brcm_vmware_ioctl_req req;
	int rc = 0;
	u32 val;

	if (copy_from_user(&req, useraddr, sizeof(req))) {
		netdev_err(dev, "%s: could not copy "
				"from user tg3_ioctl_req\n", __func__);
		return -EFAULT;
	}

	switch (req.cmd) {
	case BRCM_VMWARE_CIM_CMD_ENABLE_NIC:
		netdev_info(dev, "%s: enable NIC\n", __func__);

		rc = tg3_open(tp->dev);
		break;
	case BRCM_VMWARE_CIM_CMD_DISABLE_NIC:
		netdev_info(dev, "%s: disable NIC\n", __func__);

		rc = tg3_close(tp->dev);
		break;
	case BRCM_VMWARE_CIM_CMD_REG_READ: {
		struct brcm_vmware_ioctl_reg_read_req *rd_req;

		rd_req = &req.cmd_req.reg_read_req;
		if (0x7ffc < rd_req->reg_offset) {
			netdev_err(dev, "%s: %s: "
					"out of range: req reg: 0x%x\n",
				   __func__, "reg read", rd_req->reg_offset);
			rc = -EINVAL;
			break;
		}

		if (rd_req->reg_offset & 0x3) {
			netdev_err(dev, "%s: %s: "
				   "offset not dword aligned: req reg: 0x%x\n",
				   __func__, "reg read", rd_req->reg_offset);
			rc = -EINVAL;
			break;
		}

		switch (rd_req->reg_access_type) {
		case BRCM_VMWARE_REG_ACCESS_DIRECT:
			val = tr32(rd_req->reg_offset);
			break;
		case BRCM_VMWARE_REG_ACCESS_PCI_CFG:
			pci_read_config_dword(tp->pdev,
					      rd_req->reg_offset, &val);
			break;
		default:
			netdev_err(dev, "%s: %s: "
				  "invalid access method: access type: 0x%x\n",
				   __func__, "reg read",
				   rd_req->reg_access_type);
			rc = -EINVAL;
			break;
		}

		req.cmd_req.reg_read_req.reg_value = val;
		netdev_info(dev, "%s: %s: reg: 0x%x value:0x%x",
			    __func__, "reg read", rd_req->reg_offset,
			    rd_req->reg_value);

		break;
	} case BRCM_VMWARE_CIM_CMD_REG_WRITE: {
		struct brcm_vmware_ioctl_reg_write_req *wr_req;

		wr_req = &req.cmd_req.reg_write_req;
		if (0x7ffc < wr_req->reg_offset) {
			netdev_err(dev, "%s: %s: "
					"out of range: req reg: 0x%x\n",
				   __func__, "reg write", wr_req->reg_offset);
			rc = -EINVAL;
			break;
		}

		if (wr_req->reg_offset & 0x3) {
			netdev_err(dev, "%s: %s: "
				   "offset not dword aligned: req reg: 0x%x\n",
				   __func__, "reg write", wr_req->reg_offset);
			rc = -EINVAL;
			break;
		}

		switch (wr_req->reg_access_type) {
		case BRCM_VMWARE_REG_ACCESS_DIRECT:
			tw32(wr_req->reg_offset, wr_req->reg_value);
			break;
		case BRCM_VMWARE_REG_ACCESS_PCI_CFG:
			pci_write_config_dword(tp->pdev, wr_req->reg_offset,
					       wr_req->reg_value);
			break;
		default:
			netdev_err(dev, "%s: %s: "
				  "invalid access method: access type: 0x%x\n",
				   __func__, "reg write",
				   wr_req->reg_access_type);
			rc = -EINVAL;
			break;
		}

		netdev_info(dev, "%s: %s: reg: 0x%x value:0x%x",
			    __func__, "reg write", wr_req->reg_offset,
			    wr_req->reg_value);

		break;
	} case BRCM_VMWARE_CIM_CMD_GET_NIC_PARAM:
		netdev_info(dev, "%s: get NIC param\n", __func__);

		req.cmd_req.get_nic_param_req.mtu = dev->mtu;
		memcpy(req.cmd_req.get_nic_param_req.current_mac_addr,
		       dev->dev_addr,
		       sizeof(req.cmd_req.get_nic_param_req.current_mac_addr));
		break;
	case BRCM_VMWARE_CIM_CMD_GET_NIC_STATUS:
		netdev_info(dev, "%s: get NIC status\n", __func__);

		req.cmd_req.get_nic_status_req.nic_status = netif_running(dev);
		break;
	default:
		netdev_err(dev, "%s: unknown req.cmd: 0x%x\n",
			   __func__, req.cmd);
		rc = -EINVAL;
	}

	if (rc == 0 &&
	    copy_to_user(useraddr, &req, sizeof(req))) {
		netdev_err(dev, "%s: couldn't copy to user tg3_ioctl_req\n",
			   __func__);
		return -EFAULT;
	}

	return rc;
}
#endif  /* TG3_VMWARE_BMAPILNX */

#ifdef TG3_VMWARE_NETQ_ENABLE
static void tg3_set_prod_bdinfo(struct tg3 *tp, u32 bdinfo_addr,
				dma_addr_t mapping, u32 maxlen_flags)
{
	tw32(bdinfo_addr + TG3_BDINFO_HOST_ADDR + TG3_64BIT_REG_HIGH,
	     ((u64) mapping >> 32));
	tw32(bdinfo_addr + TG3_BDINFO_HOST_ADDR + TG3_64BIT_REG_LOW,
	     ((u64) mapping & 0xffffffff));
	tw32(bdinfo_addr + TG3_BDINFO_MAXLEN_FLAGS, maxlen_flags);
	/* Leave the nic addr field alone */
}

static void tg3_disable_prod_rcbs(struct tg3 *tp, u32 ring)
{
	struct tg3_rx_prodring_set *tpr;
	u32 offset;

	if (!(tp->tg3_flags3 & TG3_FLG3_IOV_CAPABLE))
		return;

	tpr = &tp->napi[ring].prodring;
	offset = RCVDBDI_JMB_BD_RING1 + (ring - 1) * 2 * TG3_BDINFO_SIZE;

	/* Disable the jumbo ring */
	tg3_set_prod_bdinfo(tp, offset, 0, BDINFO_FLAGS_DISABLED);
	offset += TG3_BDINFO_SIZE;

	/* Disable the standard ring */
	tg3_set_prod_bdinfo(tp, offset, 0, BDINFO_FLAGS_DISABLED);
}

static void tg3_setup_prod_rcbs(struct tg3 *tp, u32 ring)
{
	struct tg3_rx_prodring_set *tpr;
	u32 offset;

	if (!(tp->tg3_flags3 & TG3_FLG3_ENABLE_IOV))
		return;

	tpr = &tp->napi[ring].prodring;
	offset = RCVDBDI_JMB_BD_RING1 + (ring - 1) * 2 * TG3_BDINFO_SIZE;

	if (tp->tg3_flags & TG3_FLAG_JUMBO_RING_ENABLE) {
		tg3_set_prod_bdinfo(tp, offset, tpr->rx_jmb_mapping,
			      (RX_JUMBO_MAX_SIZE << BDINFO_FLAGS_MAXLEN_SHIFT) |
			      BDINFO_FLAGS_USE_EXT_RECV);
		tpr->rx_jmb_prod_idx = tp->rx_jumbo_pending;
		tw32_rx_mbox(tpr->rx_jmb_mbox, tpr->rx_jmb_prod_idx);
	}

	offset += TG3_BDINFO_SIZE;

	tg3_set_prod_bdinfo(tp, offset, tpr->rx_std_mapping,
	(RX_STD_MAX_SIZE_5717 << BDINFO_FLAGS_MAXLEN_SHIFT) |
		(TG3_RX_STD_DMA_SZ << 2));
	tpr->rx_std_prod_idx = tp->rx_pending;
	tw32_rx_mbox(tpr->rx_std_mbox, tpr->rx_std_prod_idx);
}

static void tg3_setup_prod_mboxes(struct tg3 *tp, u32 ring)
{
	struct tg3_rx_prodring_set *tpr = &tp->napi[ring].prodring;

	if (!ring) {
		tpr->rx_std_mbox = TG3_RX_STD_PROD_IDX_REG;
		tpr->rx_jmb_mbox = TG3_RX_JMB_PROD_IDX_REG;
		return;
	}

	tpr->rx_std_mbox = MAILBOX_RCV_STD_PROD_IDX_RING1 + (ring - 1) * 4;
	if (ring % 2)
		tpr->rx_std_mbox -= 4;
	else
		tpr->rx_std_mbox += 4;

	if (ring < 12)
		tpr->rx_jmb_mbox = MAILBOX_RCV_JUMBO_PROD_IDX_RING1 +
				   (ring - 1) * 4;
	else
		tpr->rx_jmb_mbox = MAILBOX_RCV_JMB_PROD_IDX_RING12 +
				   (ring - 12) * 4;

	if (ring % 2)
		tpr->rx_jmb_mbox -= 4;
	else
		tpr->rx_jmb_mbox += 4;
}

static int
tg3_netq_get_netqueue_features(vmknetddi_queueop_get_features_args_t *args)
{
	args->features = VMKNETDDI_QUEUEOPS_FEATURE_RXQUEUES;
	/* args->features |= VMKNETDDI_QUEUEOPS_FEATURE_TXQUEUES; */
	return VMKNETDDI_QUEUEOPS_OK;
}

static int
tg3_netq_get_queue_count(vmknetddi_queueop_get_queue_count_args_t *args)
{
	struct tg3 *tp = netdev_priv(args->netdev);

	if (args->type == VMKNETDDI_QUEUEOPS_QUEUE_TYPE_RX) {
		args->count = TG3_VRQ_MAX_NUM_RX_QS(tp);

		netdev_info(tp->dev, "Using %d RX NetQ rings\n", args->count);

		return VMKNETDDI_QUEUEOPS_OK;
	} else if (args->type == VMKNETDDI_QUEUEOPS_QUEUE_TYPE_TX) {
		args->count = TG3_VRQ_MAX_NUM_TX_QS(tp);

		netdev_info(tp->dev, "Using %d TX NetQ rings\n", args->count);

		return VMKNETDDI_QUEUEOPS_OK;
	} else {
		netdev_err(tp->dev, "Counting queue: invalid queue type\n");

		return VMKNETDDI_QUEUEOPS_ERR;
	}
}

static int
tg3_netq_get_filter_count(vmknetddi_queueop_get_filter_count_args_t *args)
{
	/* Only support 1 Mac filter per queue */
	args->count = 1;
	return VMKNETDDI_QUEUEOPS_OK;
}

static int
tg3_netq_alloc_tx_queue(struct net_device *netdev,
			 vmknetddi_queueops_queueid_t *p_qid,
			 u16 *queue_mapping)
{
	struct tg3 *tp = netdev_priv(netdev);
#if 0
	int i;

	if (tp->vmware.netq.n_tx_queues_allocated >= TG3_VRQ_MAX_NUM_TX_QS(tp))
		return VMKNETDDI_QUEUEOPS_ERR;

	for (i = 1; i < TG3_VRQ_MAX_NUM_TX_QS(tp) + 1; i++) {
		struct tg3_napi *tnapi = &tp->napi[i];
		if (!(tnapi->netq.flags & TG3_NETQ_TXQ_ALLOCATED)) {
			tnapi->netq.flags |= TG3_NETQ_TXQ_ALLOCATED;
			tp->vmware.netq.n_tx_queues_allocated++;
			*p_qid = VMKNETDDI_QUEUEOPS_MK_TX_QUEUEID(i);
			*queue_mapping = i;

			netdev_info(tp->dev, "TX NetQ allocated on %d\n", i);
			return VMKNETDDI_QUEUEOPS_OK;
		}
	}
#endif

	netdev_err(tp->dev, "No free tx queues found!\n");
	return VMKNETDDI_QUEUEOPS_ERR;
}

static int
tg3_netq_alloc_rx_queue(struct net_device *netdev,
			 vmknetddi_queueops_queueid_t *p_qid,
			 struct napi_struct **napi_p)
{
	int i;
	struct tg3 *tp = netdev_priv(netdev);

	if (tp->vmware.netq.n_rx_queues_allocated >= TG3_VRQ_MAX_NUM_RX_QS(tp))
		return VMKNETDDI_QUEUEOPS_ERR;

	for (i = 1; i < TG3_VRQ_MAX_NUM_RX_QS(tp) + 1; i++) {
		struct tg3_napi *tnapi = &tp->napi[i];
		if (!(tnapi->netq.flags & TG3_NETQ_RXQ_ALLOCATED)) {
			tnapi->netq.flags |= TG3_NETQ_RXQ_ALLOCATED;
			tp->vmware.netq.n_rx_queues_allocated++;
			*p_qid = VMKNETDDI_QUEUEOPS_MK_RX_QUEUEID(i);
			*napi_p = &tnapi->napi;

			netdev_info(tp->dev, "RX NetQ allocated on %d\n", i);
			return VMKNETDDI_QUEUEOPS_OK;
		}
	}
	netdev_err(tp->dev, "No free rx queues found!\n");
	return VMKNETDDI_QUEUEOPS_ERR;
}

static int
tg3_netq_alloc_queue(vmknetddi_queueop_alloc_queue_args_t *args)
{
	struct net_device *netdev = args->netdev;
	struct tg3 *tp = netdev_priv(netdev);

	if (args->type == VMKNETDDI_QUEUEOPS_QUEUE_TYPE_TX) {
		return tg3_netq_alloc_tx_queue(args->netdev, &args->queueid,
					     &args->queue_mapping);
	} else if (args->type == VMKNETDDI_QUEUEOPS_QUEUE_TYPE_RX) {
		return tg3_netq_alloc_rx_queue(args->netdev, &args->queueid,
						&args->napi);
	} else {
		netdev_err(tp->dev, "Trying to alloc invalid queue type: %x\n",
			   args->type);
		return VMKNETDDI_QUEUEOPS_ERR;
	}
}

static void tg3_netq_txq_free(struct tg3 *tp, struct tg3_napi *tnapi)
{
	int rc;

	tnapi->netq.flags &= ~TG3_NETQ_TXQ_ALLOCATED;
	tp->vmware.netq.n_tx_queues_allocated--;

	/* Wait for the TX to flush all packets */
	/* NOTE : Assumes no more packets will be submitted to this queue. */
	tnapi->netq.flags |= TG3_NETQ_TXQ_FREE_STATE;
	rc = wait_event_timeout(tp->vmware.netq.tx_wait,
				!(tnapi->netq.flags &
				  TG3_NETQ_TXQ_FREE_STATE),
				TG3_NETQ_WAIT_EVENT_TIMEOUT);
	tnapi->netq.flags &= ~TG3_NETQ_TXQ_FREE_STATE;
}

static int
tg3_netq_free_tx_queue(struct net_device *netdev,
		       vmknetddi_queueops_queueid_t qid)
{
	struct tg3 *tp = netdev_priv(netdev);
	struct tg3_napi *tnapi;

	u16 index = VMKNETDDI_QUEUEOPS_QUEUEID_VAL(qid);
	if (index == 0 || index > TG3_VRQ_MAX_NUM_TX_QS(tp))
		return VMKNETDDI_QUEUEOPS_ERR;

	tnapi = &tp->napi[index];

	if (!(tnapi->netq.flags & TG3_NETQ_TXQ_ALLOCATED))
		return VMKNETDDI_QUEUEOPS_ERR;

	tg3_netq_txq_free(tp, tnapi);

	if (tnapi->tx_cons != tnapi->tx_prod) {
		netdev_warn(tp->dev,
			    "Timeout submitting free NetQ TX Queue: %x\n",
			    index);
		return VMKNETDDI_QUEUEOPS_ERR;
	}

	netdev_info(tp->dev, "Free NetQ TX Queue: %x\n", index);

	return VMKNETDDI_QUEUEOPS_OK;
}

static void
tg3_netq_disable_queue(struct tg3 *tp, int qid)
{
	int i, rc;
	struct tg3_napi *tnapi = &tp->napi[qid];
	u32 val;

	tg3_disable_prod_rcbs(tp, qid);

	/* Disable the VRQ */
	tw32(MAC_VRQ_ENABLE, tr32(MAC_VRQ_ENABLE) & ~(1 << qid));

	/* Poll for acknowledgement from the hardware. */
	for (i = 0; i < TG3_VRQ_CHANGE_TIMEOUT_US / 10; i++) {
		if (!(tr32(VRQ_STATUS) & (1 << qid)))
			break;
		udelay(10);
	}

	if (i == TG3_VRQ_CHANGE_TIMEOUT_US / 10)
		netdev_warn(tp->dev,
			    "Timeout performing initial queue disable: %x\n",
			    qid);

	/* Remove the perfect match filter. */
	if (qid < 4) {
		tw32(MAC_ADDR_0_HIGH + (qid * 8), tr32(MAC_ADDR_0_HIGH));
		tw32(MAC_ADDR_0_LOW + (qid * 8), tr32(MAC_ADDR_0_LOW));
	} else {
		tw32(MAC_VRQ_PMATCH_HI_5 + (qid - 4) * 8,
		     tr32(MAC_ADDR_0_HIGH));
		tw32(MAC_VRQ_PMATCH_LO_5 + (qid - 4) * 8,
		     tr32(MAC_ADDR_0_LOW));
	}

	/* Force a DMA of all remaining rx packets in this VRQ */
	tw32(HOSTCC_MODE,
	     tp->coalesce_mode | HOSTCC_MODE_ENABLE | tnapi->coal_now);

	/* Wait for ISR to ack */
	tnapi->netq.flags |= TG3_NETQ_RXQ_FREE_STATE;
	rc = wait_event_timeout(tp->vmware.netq.rx_wait,
				!(tnapi->netq.flags &
				  TG3_NETQ_RXQ_FREE_STATE),
				TG3_NETQ_WAIT_EVENT_TIMEOUT);
	tnapi->netq.flags &= ~TG3_NETQ_RXQ_FREE_STATE;

	tw32(HOSTCC_PARAM_SET_RESET, 1 << qid);

	tw32_rx_mbox(tnapi->prodring.rx_std_mbox, 0);
	tw32_rx_mbox(tnapi->prodring.rx_jmb_mbox, 0);
	tw32_rx_mbox(tnapi->consmbox, 0);

	tw32(HOSTCC_RXCOL_TICKS_VEC1 + (qid - 1) * 0x18, 0);
	tw32(HOSTCC_RXMAX_FRAMES_VEC1 + (qid - 1) * 0x18, 0);
	tw32(HOSTCC_RXCOAL_MAXF_INT_VEC1 + (qid - 1) * 0x18, 0);

	val = VRQ_FLUSH_ENABLE | VRQ_FLUSH_RESET_ENABLE |
	      VRQ_FLUSH_STATUPDT_INT_ENABLE | VRQ_FLUSH_DISCARD_PKT_ENABLE |
	      VRQ_FLUSH_SW_FLUSH;
	tw32(VRQ_FLUSH_CTRL, val | (1 << (qid + 15)));

	/* Poll for acknowledgement from the hardware. */
	for (i = 0; i < TG3_VRQ_FLUSH_TIMEOUT_US / 10; i++) {
		if (!(tr32(VRQ_FLUSH_CTRL) & VRQ_FLUSH_SW_FLUSH))
			break;
		udelay(10);
	}
	if (tr32(VRQ_FLUSH_CTRL) & VRQ_FLUSH_SW_FLUSH) {
		netdev_warn(tp->dev,
			    "Timeout flushing hardware queue: %x\n",
			    qid);
	}

	tg3_rx_prodring_free(tp, &tnapi->prodring);

	tnapi->netq.flags &= ~TG3_NETQ_RXQ_ENABLED;
}

static int
tg3_netq_free_rx_queue(struct net_device *netdev,
		       vmknetddi_queueops_queueid_t qid)
{
	struct tg3 *tp = netdev_priv(netdev);
	struct tg3_napi *tnapi;

	u16 index = VMKNETDDI_QUEUEOPS_QUEUEID_VAL(qid);
	if (index == 0 || index > TG3_VRQ_MAX_NUM_RX_QS(tp))
		return VMKNETDDI_QUEUEOPS_ERR;

	tnapi = &tp->napi[index];
	if (!(tnapi->netq.flags & TG3_NETQ_RXQ_ALLOCATED))
		return VMKNETDDI_QUEUEOPS_ERR;

	if (tnapi->netq.flags & TG3_NETQ_RXQ_ENABLED)
		tg3_netq_disable_queue(tp, index);

	tnapi->netq.flags &= ~TG3_NETQ_RXQ_ALLOCATED;
	tp->vmware.netq.n_rx_queues_allocated--;

	netdev_info(tp->dev, "Free NetQ RX Queue: %x\n", index);

	return VMKNETDDI_QUEUEOPS_OK;
}

static int
tg3_netq_free_queue(vmknetddi_queueop_free_queue_args_t *args)
{
	struct tg3 *tp = netdev_priv(args->netdev);
	if (VMKNETDDI_QUEUEOPS_IS_TX_QUEUEID(args->queueid)) {
		return tg3_netq_free_tx_queue(args->netdev, args->queueid);
	} else if (VMKNETDDI_QUEUEOPS_IS_RX_QUEUEID(args->queueid)) {
		return tg3_netq_free_rx_queue(args->netdev, args->queueid);
	} else {
		netdev_err(tp->dev, "free netq: invalid queue type\n");
		return VMKNETDDI_QUEUEOPS_ERR;
	}
}

static int
tg3_netq_get_queue_vector(vmknetddi_queueop_get_queue_vector_args_t *args)
{
	int qid;
	struct tg3 *tp = netdev_priv(args->netdev);
	qid = VMKNETDDI_QUEUEOPS_QUEUEID_VAL(args->queueid);
	if (qid > tp->irq_cnt - 1)
		return VMKNETDDI_QUEUEOPS_ERR;

	args->vector = tp->napi[qid].irq_vec;

	return VMKNETDDI_QUEUEOPS_OK;
}

static int
tg3_netq_get_default_queue(vmknetddi_queueop_get_default_queue_args_t *args)
{
	struct tg3 *tp = netdev_priv(args->netdev);

	if (args->type == VMKNETDDI_QUEUEOPS_QUEUE_TYPE_RX) {
		args->queueid = VMKNETDDI_QUEUEOPS_MK_RX_QUEUEID(0);
		args->napi = &tp->napi[0].napi;
		return VMKNETDDI_QUEUEOPS_OK;
	} else if (args->type == VMKNETDDI_QUEUEOPS_QUEUE_TYPE_TX) {
		args->queueid = VMKNETDDI_QUEUEOPS_MK_TX_QUEUEID(0);
		args->queue_mapping = 0;
		return VMKNETDDI_QUEUEOPS_OK;
	} else
		return VMKNETDDI_QUEUEOPS_ERR;
}

static int
tg3_netq_remove_rx_filter(vmknetddi_queueop_remove_rx_filter_args_t *args)
{
	struct tg3 *tp = netdev_priv(args->netdev);
	struct tg3_napi *tnapi;
	u16 qid = VMKNETDDI_QUEUEOPS_QUEUEID_VAL(args->queueid);
	u16 fid = VMKNETDDI_QUEUEOPS_FILTERID_VAL(args->filterid);

	if (!VMKNETDDI_QUEUEOPS_IS_RX_QUEUEID(args->queueid))
		return VMKNETDDI_QUEUEOPS_ERR;

	if (qid == 0 || qid > TG3_VRQ_MAX_NUM_RX_QS(tp))
		return VMKNETDDI_QUEUEOPS_ERR;

	tnapi = &tp->napi[qid];

	/* Only support one Mac filter per queue */
	if (fid != 0 ||
	    !(tnapi->netq.flags & TG3_NETQ_RXQ_ENABLED))
		return VMKNETDDI_QUEUEOPS_ERR;

	tg3_netq_disable_queue(tp, qid);

	netdev_info(tp->dev, "NetQ remove RX filter: %d\n", qid);

	return VMKNETDDI_QUEUEOPS_OK;
}

static int
tg3_netq_apply_rx_filter(vmknetddi_queueop_apply_rx_filter_args_t *args)
{
	u8 *macaddr;
	struct tg3_napi *tnapi;
	struct tg3 *tp = netdev_priv(args->netdev);
	vmknetddi_queueops_filter_class_t class;
	u16 qid = VMKNETDDI_QUEUEOPS_QUEUEID_VAL(args->queueid);
	DECLARE_MAC_BUF(mac);
	u32 val;

	if (!VMKNETDDI_QUEUEOPS_IS_RX_QUEUEID(args->queueid)) {
		netdev_err(tp->dev, "Invalid NetQ RX ID: %x\n", args->queueid);
		return VMKNETDDI_QUEUEOPS_ERR;
	}

	class = vmknetddi_queueops_get_filter_class(&args->filter);
	if ((class != VMKNETDDI_QUEUEOPS_FILTER_MACADDR) &&
	    (class != VMKNETDDI_QUEUEOPS_FILTER_VLANMACADDR)) {
		netdev_err(tp->dev, "Received invalid RX NetQ filter: %x\n",
			   class);
		return VMKNETDDI_QUEUEOPS_ERR;
	}

	if (qid == 0 || qid > TG3_VRQ_MAX_NUM_RX_QS(tp)) {
		netdev_err(tp->dev,
			   "Applying filter with invalid RX NetQ %d ID\n", qid);
		return VMKNETDDI_QUEUEOPS_ERR;
	}

	tnapi = &tp->napi[qid];

	if (!(tnapi->netq.flags & TG3_NETQ_RXQ_ALLOCATED)) {
		netdev_err(tp->dev, "RX NetQ %d not allocated\n", qid);
		return VMKNETDDI_QUEUEOPS_ERR;
	}

	if (tnapi->netq.flags & TG3_NETQ_RXQ_ENABLED) {
		netdev_err(tp->dev, "RX NetQ %d already enabled\n", qid);
		return VMKNETDDI_QUEUEOPS_ERR;
	}

	/*  Apply RX filter code here */
	args->filterid = VMKNETDDI_QUEUEOPS_MK_FILTERID(qid);

	macaddr = (void *)vmknetddi_queueops_get_filter_macaddr(&args->filter);

	/* Populate the producer rings with skbs */
	if (tg3_rx_prodring_alloc(tp, &tnapi->prodring)) {
		netdev_err(tp->dev, "Failed to allocate queue buffers!\n");
		return VMKNETDDI_QUEUEOPS_ERR;
	}

	/* Make the producer rings available to the hardware */
	tg3_setup_prod_rcbs(tp, qid);

	/* Program the perfect match filter. */
	if (qid < 4) {
		val = (macaddr[0] << 8) | macaddr[1];
		tw32(MAC_ADDR_0_HIGH + (qid * 8), val);

		val = ((macaddr[2] << 24) | (macaddr[3] << 16) |
		       (macaddr[4] <<  8) | (macaddr[5] <<  0));
		tw32(MAC_ADDR_0_LOW + (qid * 8), val);
	} else {
		val = (macaddr[0] << 8)  | macaddr[1];
		tw32(MAC_VRQ_PMATCH_HI_5 + (qid - 4) * 8, val);

		val = (macaddr[2] << 24) | (macaddr[3] << 16) |
			  (macaddr[4] <<  8) | (macaddr[5] <<  0);
		tw32(MAC_VRQ_PMATCH_LO_5 + (qid - 4) * 8, val);
	}

	/* Tell the hardware which perfect match filter to use. */
	val = MAC_VRQMAP_1H_PTA_PFEN | qid;
	tw32(MAC_VRQMAP_1H + qid * 8, val);
	val = MAC_VRQMAP_2H_PTA_OR | MAC_VRQMAP_2H_PTA_EN;
	tw32(MAC_VRQMAP_2H + qid * 8, val);

	/* Enable the VRQ */
	val = tr32(MAC_VRQ_ENABLE);
	tw32(MAC_VRQ_ENABLE, val | (1 << qid));

	tw32(HOSTCC_RXCOL_TICKS_VEC1 + (qid - 1) * 0x18,
	     tp->coal.rx_coalesce_usecs);
	tw32(HOSTCC_RXMAX_FRAMES_VEC1 + (qid - 1) * 0x18,
	     tp->coal.rx_max_coalesced_frames);
	tw32(HOSTCC_RXCOAL_MAXF_INT_VEC1 + (qid - 1) * 0x18,
	     tp->coal.rx_max_coalesced_frames_irq);

	tnapi->netq.flags |= TG3_NETQ_RXQ_ENABLED;

	netdev_info(tp->dev, "NetQ set RX Filter: %d [%s %d]\n",
		    qid, print_mac(mac, macaddr),
		    vmknetddi_queueops_get_filter_vlanid(&args->filter));

	return VMKNETDDI_QUEUEOPS_OK;
}

static int
tg3_netq_get_queue_stats(vmknetddi_queueop_get_stats_args_t *args)
{
	u16 qid = VMKNETDDI_QUEUEOPS_QUEUEID_VAL(args->queueid);
	struct tg3_napi *tnapi;
	struct tg3 *tp = netdev_priv(args->netdev);
	struct net_device_stats *netstats;

	tnapi = &tp->napi[qid];
	netstats = &tnapi->netq.net_stats;

	netstats->rx_packets = tnapi->netq.stats.rx_packets;
	netstats->rx_bytes = tnapi->netq.stats.rx_bytes;
	netstats->rx_errors = tnapi->netq.stats.rx_errors;
	netstats->rx_crc_errors = tnapi->netq.stats.rx_crc_errors;
	netstats->rx_frame_errors = tnapi->netq.stats.rx_frame_errors;
	netstats->tx_packets = tnapi->netq.stats.tx_packets;
	netstats->tx_bytes = tnapi->netq.stats.tx_bytes;

	args->stats = netstats;

	return VMKNETDDI_QUEUEOPS_OK;
}

static int
tg3_netqueue_ops(vmknetddi_queueops_op_t op, void *args)
{
	struct tg3 *tp;

	if (op == VMKNETDDI_QUEUEOPS_OP_GET_VERSION)
		return vmknetddi_queueops_version(
			(vmknetddi_queueop_get_version_args_t *)args);

	tp = netdev_priv(((vmknetddi_queueop_get_features_args_t *)args)->netdev);

	if (!(tp->tg3_flags2 & TG3_FLG2_USING_MSIX) ||
	    !(tp->tg3_flags & TG3_FLAG_INIT_COMPLETE) ||
	    tp->irq_cnt < 2)
		return VMKNETDDI_QUEUEOPS_ERR;

	switch (op) {
	case VMKNETDDI_QUEUEOPS_OP_GET_FEATURES:
		return tg3_netq_get_netqueue_features(
			(vmknetddi_queueop_get_features_args_t *)args);

	case VMKNETDDI_QUEUEOPS_OP_GET_QUEUE_COUNT:
		return tg3_netq_get_queue_count(
			(vmknetddi_queueop_get_queue_count_args_t *)args);

	case VMKNETDDI_QUEUEOPS_OP_GET_FILTER_COUNT:
		return tg3_netq_get_filter_count(
			(vmknetddi_queueop_get_filter_count_args_t *)args);

	case VMKNETDDI_QUEUEOPS_OP_ALLOC_QUEUE:
		return tg3_netq_alloc_queue(
			(vmknetddi_queueop_alloc_queue_args_t *)args);

	case VMKNETDDI_QUEUEOPS_OP_FREE_QUEUE:
		return tg3_netq_free_queue(
			(vmknetddi_queueop_free_queue_args_t *)args);

	case VMKNETDDI_QUEUEOPS_OP_GET_QUEUE_VECTOR:
		return tg3_netq_get_queue_vector(
			(vmknetddi_queueop_get_queue_vector_args_t *)args);

	case VMKNETDDI_QUEUEOPS_OP_GET_DEFAULT_QUEUE:
		return tg3_netq_get_default_queue(
			(vmknetddi_queueop_get_default_queue_args_t *)args);

	case VMKNETDDI_QUEUEOPS_OP_APPLY_RX_FILTER:
		return tg3_netq_apply_rx_filter(
			(vmknetddi_queueop_apply_rx_filter_args_t *)args);

	case VMKNETDDI_QUEUEOPS_OP_REMOVE_RX_FILTER:
		return tg3_netq_remove_rx_filter(
			(vmknetddi_queueop_remove_rx_filter_args_t *)args);

	case VMKNETDDI_QUEUEOPS_OP_GET_STATS:
		return tg3_netq_get_queue_stats(
			(vmknetddi_queueop_get_stats_args_t *)args);

	/*  Unsupported for now */
	default:
		break;
	}

	netdev_warn(to->dev, "Unhandled NETQUEUE OP %d\n", op);

	return VMKNETDDI_QUEUEOPS_ERR;
}

static void tg3_netq_init(struct tg3 *tp)
{
	int i;
	struct tg3_netq_napi *tnetq;

	if (!(tp->tg3_flags3 & TG3_FLG3_IOV_CAPABLE)) {
		for (i = 0; i < 5; i++)
			tp->napi[i].srcprodring = &tp->napi[0].prodring;
		return;
	}

	for (i = 0; i < tp->irq_max; i++)
		tp->napi[i].srcprodring = &tp->napi[i].prodring;

	tnetq = &tp->napi[0].netq;
	tnetq->flags |= TG3_NETQ_RXQ_ALLOCATED;

	tnetq->flags |= TG3_NETQ_TXQ_ALLOCATED;
	tnetq = &tp->napi[1].netq;
	tnetq->flags |= TG3_NETQ_TXQ_ALLOCATED;

	init_waitqueue_head(&tp->vmware.netq.rx_wait);
	init_waitqueue_head(&tp->vmware.netq.tx_wait);

	VMKNETDDI_REGISTER_QUEUEOPS(tp->dev, tg3_netqueue_ops);
	netdev_info(tp->dev, "VMware NetQueue Ops is registered\n");
}

static void tg3_netq_free_all_qs(struct tg3 *tp)
{
	int i;

	for (i = 1; i < tp->irq_max; i++) {
		struct tg3_napi *tnapi = &tp->napi[i];

		if (i != 1 && (tnapi->netq.flags & TG3_NETQ_TXQ_ALLOCATED))
			tg3_netq_txq_free(tp, tnapi);

		if (tnapi->netq.flags & TG3_NETQ_RXQ_ALLOCATED) {
			if (tnapi->netq.flags & TG3_NETQ_RXQ_ENABLED)
				tg3_netq_disable_queue(tp, i);

			tnapi->netq.flags &= ~TG3_NETQ_RXQ_ALLOCATED;
			tp->vmware.netq.n_rx_queues_allocated--;
		}
	}
}

static void tg3_netq_invalidate_state(struct tg3 *tp)
{
	if (!(tp->tg3_flags3 & TG3_FLG3_ENABLE_IOV))
		return;

	tg3_netq_free_all_qs(tp);
	vmknetddi_queueops_invalidate_state(tp->dev);
}

static void tg3_netq_restore(struct tg3 *tp)
{
	if (!(tp->tg3_flags3 & TG3_FLG3_IOV_CAPABLE))
		return;

	/* Enable the VRQs */
	tw32(MAC_VRQ_ENABLE, MAC_VRQ_ENABLE_DFLT_VRQ);
}
#endif /* TG3_VMWARE_NETQ_ENABLE */
