/*
 * sngtc_client
 * Sangoma Transcoder SOAP Client Utility
 *
 * Moises Silva <moy@sangoma.com>
 *
 * Copyright (C) 2010 Sangoma Technologies
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * * Neither the name of the original author; nor the names of any contributors
 * may be used to endorse or promote products derived from this software
 * without specific prior written permission.
 * 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Contributors:
 *
 */

#define _GNU_SOURCE

#include <ortp/ortp.h>
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <poll.h>
#include <sys/syscall.h>

#ifdef SNGTC_LIB_BIND
#include <sng_tc/sng_tc.h>
#else
#include <sng_tc/sngtc_node.h>
#endif

#define SNGTC_LOGLEVEL_BERT (SNGTC_LOGLEVEL_PUBLIC + 1)
#define DEFAULT_BASE_UDP_PORT 10000
#define INTERVAL_STEP 10
#define MILLISECONDS_IN_SECOND 1000

#define VERBOSITY_NONE 0
#define VERBOSITY_ONLY_ERRORS 1
#define VERBOSITY_ALL 2

static int local_sngtc_logger(int level, char *fmt, ...);

static sngtc_init_cfg_t g_init_cfg;

typedef struct {
	int bytes;
	int time;
} codec_interval_t;

#define CODEC_FLAG_BYTESWAP (1 << 0)
typedef struct {
	/* User name for the codec */
	const char *name;
	/* Description for the codec */
	const char *desc;
	/* Sangoma codec id */
	enum sngtc_codec_definition codec_id;
	/* RTP payload (IANA) */
	int payload;
	/* default ms interval for this codec if none specified */
	int default_interval;
	/* how many bytes per 10ms interval for this codec */
	int per_interval_bytes;
	/* Maximum supported interval */
	int max_interval;
	/* Minimum supported interval */
	int min_interval;
	/* sampling rate */
	int sampling;
	/* array of fixed supported intervals if any 
	 * (if this is set, max, min and per_interval bytes members are not used) */
	codec_interval_t fixed_intervals[10];
	/* misc flags */
	uint32_t flags;
} codec_t;

static struct globals_s {
	/* program running */
	volatile uint8_t running;

	/* init configuration structure for libsngtc */
	sngtc_init_cfg_t init_cfg;

	/* SOAP address to contact */
	char soap_addr_str[512];

	/* Input file */
	char in_file_str[512];

	/* Output file string */
	char out_file_str[512];

	/* RTP ptime for the input codec */
	int in_interval;
	int in_bytes;

	/* RTP ptime for the output codec */
	int out_interval;
	int out_bytes;

	/* base starting port for RTP connections */
	short base_port;

	char bindip_str[255];
	int32_t bindip_addr;

	/* input codec selected */
	codec_t *in_codec;

	/* output codec selected */
	codec_t *out_codec;

	/* process ID */
	int pid;
	
	/* RTP debug enabled */
	uint8_t rtpdebug;

	/* How much output to print */
	uint8_t verbosity;

	/* write a wav file when writing slinear */
	uint8_t write_wav;

	/* do BERT testing (only valid for pcmu codec)*/
	uint8_t do_bert;
	
	/* stop test after timeout seconds. if 0 do test until ctrl-c */
	uint32_t timeout;

	/* Do not destroy sessions until all sessions finish streaming */
	uint8_t linger;
	
	/* Do not print anything */
	uint8_t bert_silent;

	/* number of sessions to launch */
	int sessions;

	/* mutex to synchronize access to some globals */
	pthread_mutex_t the_mutex;

	/* TLS for the thread id */
	pthread_key_t thread_id_key;

	/* how many threads are currently streaming */
	volatile int sessions_streaming;

	/* how many threads are currently running */
	volatile int sessions_running;
	
	volatile int sessions_sync;

	/* errors */
	int32_t errors;

	/* global rx packets */
	int32_t rxpackets;

	int32_t sync_err;
} globals;


/* list of codecs we support see http://www.iana.org/assignments/rtp-parameters for the list of payload IANA values */
static codec_t global_codecs[] = {
	{
		.name = "L16_1",
		.desc = "Signed Linear 16bit (128kbps) (8Khz)",
		.codec_id = SNGTC_CODEC_L16_1,
		.payload = IANA_L16_A_8000_1,
		.default_interval = 20,
		.per_interval_bytes = 160,
		.min_interval = 10,
		.max_interval = 50,
		.fixed_intervals = { { 0 , 0 } },
		.sampling = 8000,
		.flags = CODEC_FLAG_BYTESWAP,
	},
	{
		.name = "L16_2",
		.desc = "Signed Linear 16bit (256kbps) (16Khz)",
		.codec_id = SNGTC_CODEC_L16_2,
		.payload = IANA_L16_A_16000_1,
		.default_interval = 20,
		.per_interval_bytes = 320,
		.min_interval = 10,
		.max_interval = 50,
		.fixed_intervals = { { 0 , 0 } },
		.sampling = 16000,
		.flags = CODEC_FLAG_BYTESWAP,
	},
	{
		.name = "PCMU",
		.desc = "G.711 Ulaw (64kbps) (8Khz)",
		.codec_id = SNGTC_CODEC_PCMU,
		.payload = IANA_PCMU_A_8000_1,
		.default_interval = 20,
		.per_interval_bytes = 80,
		.min_interval = 10,
		.max_interval = 50,
		.fixed_intervals = { { 0 , 0 } },
		.sampling = 8000,
		.flags = 0,
	},
	{
		.name = "PCMA",
		.desc = "G.711 Alaw (64kbps) (8Khz)",
		.codec_id = SNGTC_CODEC_PCMA,
		.payload = IANA_PCMA_A_8000_1,
		.default_interval = 20,
		.per_interval_bytes = 80,
		.min_interval = 10,
		.max_interval = 50,
		.fixed_intervals = { { 0 , 0 } },
		.sampling = 8000,
		.flags = 0,
	},
	{
		.name = "G729AB",
		.desc = "G.729 Annex A and B (8kbps) (8Khz)",
		.codec_id = SNGTC_CODEC_G729AB,
		.payload = IANA_G729_AB_8000_1,
		.default_interval = 20,
		.per_interval_bytes = 10,
		.min_interval = 10,
		.max_interval = 200,
		.fixed_intervals = { { 0 , 0 } },
		.sampling = 8000,
		.flags = 0,
	},
	{
		.name = "GSM_FR",
		.desc = "GSM Full Rate (13kbps) (8Khz)",
		.codec_id = SNGTC_CODEC_GSM_FR,
		.payload = IANA_GSM_A_8000_1,
		.default_interval = 20,
		.per_interval_bytes = 0,
		.min_interval = 0,
		.max_interval = 0,
		.fixed_intervals = 
		{ 
			{ 
				.time = 20, 
				.bytes = 33 
			},
			{ 0 , 0 },
		},
		.sampling = 8000,
		.flags = 0,
	},
	{
		.name = "ILBC_152",
		.desc = "iLBC (15.2kbps) (8Khz)",
		.codec_id = SNGTC_CODEC_ILBC_152,
		.payload = IANA_ILBC_152_8000_1,
		.default_interval = 20,
		.per_interval_bytes = 0,
		.min_interval = 0,
		.max_interval = 0,
		.fixed_intervals = 
		{ 
			{ 
				.time = 20, 
				.bytes = 38 
			},
			{ 
				.time = 40, 
				.bytes = 76 
			},
			{ 
				.time = 60, 
				.bytes = 114 
			},
			{ 0 , 0 },
		},
		.sampling = 8000,
		.flags = 0,
	},
	{
		.name = "ILBC_133",
		.desc = "iLBC (13.3kbps) (8Khz)",
		.codec_id = SNGTC_CODEC_ILBC_133,
		.payload = IANA_ILBC_133_8000_1,
		.default_interval = 30,
		.per_interval_bytes = 0,
		.min_interval = 0,
		.max_interval = 0,
		.fixed_intervals = 
		{ 
			{ 
				.time = 30, 
				.bytes = 50 
			},
			{ 
				.time = 60, 
				.bytes = 100 
			},
			{ 0 , 0 },
		},
		.sampling = 8000,
		.flags = 0,
	},
	{
		.name = "G723_1_53",
		.desc = "G723.1 (5.3kbps) (8Khz)",
		.codec_id = SNGTC_CODEC_G723_1_53,
		.payload = IANA_G723_A_8000_1,
		.default_interval = 30,
		.per_interval_bytes = 0,
		.min_interval = 0,
		.max_interval = 0,
		.fixed_intervals = 
		{ 
			{ 
				.time = 30, 
				.bytes = 20 
			},
			{ 
				.time = 60, 
				.bytes = 40 
			},
			{ 
				.time = 90, 
				.bytes = 60 
			},
			{ 0 , 0 },
		},
		.sampling = 8000,
		.flags = 0,
	},
	{
		.name = "G723_1_63",
		.desc = "G723.1 (6.3kbps) (8Khz)",
		.codec_id = SNGTC_CODEC_G723_1_63,
		.payload = IANA_G723_A_8000_1,
		.default_interval = 30,
		.per_interval_bytes = 0,
		.min_interval = 0,
		.max_interval = 0,
		.fixed_intervals = 
		{ 
			{ 
				.time = 30, 
				.bytes = 24 
			},
			{ 
				.time = 60, 
				.bytes = 48 
			},
			{ 
				.time = 90, 
				.bytes = 72 
			},
			{ 0 , 0 },
		},
		.sampling = 8000,
		.flags = 0,
	},
	{
		.name = "G726_32",
		.desc = "G726 (32kbps) (8Khz)",
		.codec_id = SNGTC_CODEC_G726_32,
		.payload = IANA_G726_32_8000_1,
		.default_interval = 20,
		.per_interval_bytes = 40,
		.min_interval = 10,
		.max_interval = 50,
		.fixed_intervals = { { 0 , 0 } },
		.sampling = 8000,
		.flags = 0,
	},
	{
		.name = "G722",
		.desc = "G722 (64kbps) (8Khz)",
		.codec_id = SNGTC_CODEC_G722,
		.payload = IANA_G722_A_8000_1,
		.default_interval = 20,
		.per_interval_bytes = 80,
		.min_interval = 10,
		.max_interval = 40,
		.fixed_intervals = { { 0 , 0 } },
		.sampling = 8000,
		.flags = 0,
	},
	{
		.name = "SIREN7_24",
		.desc = "Siren 7 (G722.1) (24kbps) (16Khz)",
		.codec_id = SNGTC_CODEC_SIREN7_24,
		.payload = IANA_SIREN7,
		.default_interval = 20,
		.per_interval_bytes = 30,
		.min_interval = 20,
		.max_interval = 20,
		.fixed_intervals = { { 0 , 0 } },
		.sampling = 16000,
		.flags = 0,
	},
	{
		.name = "SIREN7_32",
		.desc = "Siren 7 (G722.1) (32kbps) (16Khz)",
		.codec_id = SNGTC_CODEC_SIREN7_32,
		.payload = IANA_SIREN7,
		.default_interval = 20,
		.per_interval_bytes = 40,
		.min_interval = 20,
		.max_interval = 20,
		.fixed_intervals = { { 0 , 0 } },
		.sampling = 16000,
		.flags = 0,
	},
#if 0
	{
		.name = "SIREN7_48",
		.desc = "Siren 7 (G722.1) (48kbps) (16Khz)",
		.codec_id = SNGTC_CODEC_SIREN7_48,
		.payload = IANA_SIREN7,
		.default_interval = 20,
		.per_interval_bytes = 40,
		.min_interval = 20,
		.max_interval = 20,
		.fixed_intervals = { { 0 , 0 } },
		.sampling = 16000,
		.flags = 0,
	},
#endif
	{
		.name = "AMR_475",
		.desc = "AMR (4.75kbps) (8Khz)",
		.codec_id = SNGTC_CODEC_AMR_475,
		.payload = IANA_AMR_A_8000_1,
		.default_interval = 20,
		.per_interval_bytes = 0,
		.min_interval = 0,
		.max_interval = 0,
		.fixed_intervals = 
		{ 
			{ 
				.time = 20, 
				.bytes = 14 
			},
			{ 
				.time = 40, 
				.bytes = 27 
			},
		},
		.sampling = 8000,
		.flags = 0,
	},
	{
		.name = "AMR_515",
		.desc = "AMR (5.15kbps) (8Khz)",
		.codec_id = SNGTC_CODEC_AMR_515,
		.payload = IANA_AMR_A_8000_1,
		.default_interval = 20,
		.per_interval_bytes = 0,
		.min_interval = 0,
		.max_interval = 0,
		.fixed_intervals = 
		{ 
			{ 
				.time = 20, 
				.bytes = 15 
			},
			{ 
				.time = 40, 
				.bytes = 29 
			},
		},
		.sampling = 8000,
		.flags = 0,
	},
	{
		.name = "AMR_590",
		.desc = "AMR (5.90kbps) (8Khz)",
		.codec_id = SNGTC_CODEC_AMR_590,
		.payload = IANA_AMR_A_8000_1,
		.default_interval = 20,
		.per_interval_bytes = 0,
		.min_interval = 0,
		.max_interval = 0,
		.fixed_intervals = 
		{ 
			{ 
				.time = 20, 
				.bytes = 17 
			},
			{ 
				.time = 40, 
				.bytes = 33 
			},
		},
		.sampling = 8000,
		.flags = 0,
	},
	{
		.name = "AMR_670",
		.desc = "AMR (6.70kbps) (8Khz)",
		.codec_id = SNGTC_CODEC_AMR_670,
		.payload = IANA_AMR_A_8000_1,
		.default_interval = 20,
		.per_interval_bytes = 0,
		.min_interval = 0,
		.max_interval = 0,
		.fixed_intervals = 
		{ 
			{ 
				.time = 20, 
				.bytes = 19 
			},
			{ 
				.time = 40, 
				.bytes = 37 
			},
		},
		.sampling = 8000,
		.flags = 0,
	},
	{
		.name = "AMR_740",
		.desc = "AMR (7.40kbps) (8Khz)",
		.codec_id = SNGTC_CODEC_AMR_740,
		.payload = IANA_AMR_A_8000_1,
		.default_interval = 20,
		.per_interval_bytes = 0,
		.min_interval = 0,
		.max_interval = 0,
		.fixed_intervals = 
		{ 
			{ 
				.time = 20, 
				.bytes = 21 
			},
			{ 
				.time = 40, 
				.bytes = 41 
			},
		},
		.sampling = 8000,
		.flags = 0,
	},
	{
		.name = "AMR_795",
		.desc = "AMR (7.95kbps) (8Khz)",
		.codec_id = SNGTC_CODEC_AMR_795,
		.payload = IANA_AMR_A_8000_1,
		.default_interval = 20,
		.per_interval_bytes = 0,
		.min_interval = 0,
		.max_interval = 0,
		.fixed_intervals = 
		{ 
			{ 
				.time = 20, 
				.bytes = 22 
			},
			{ 
				.time = 40, 
				.bytes = 43 
			},
		},
		.sampling = 8000,
		.flags = 0,
	},
	{
		.name = "AMR_1020",
		.desc = "AMR (10.20kbps) (8Khz)",
		.codec_id = SNGTC_CODEC_AMR_1020,
		.payload = IANA_AMR_A_8000_1,
		.default_interval = 20,
		.per_interval_bytes = 0,
		.min_interval = 0,
		.max_interval = 0,
		.fixed_intervals = 
		{ 
			{ 
				.time = 20, 
				.bytes = 28 
			},
			{ 
				.time = 40, 
				.bytes = 55 
			},
		},
		.sampling = 8000,
		.flags = 0,
	},
	{
		.name = "AMR_1220",
		.desc = "AMR (12.20kbps) (8Khz)",
		.codec_id = SNGTC_CODEC_AMR_1220,
		.payload = IANA_AMR_A_8000_1,
		.default_interval = 20,
		.per_interval_bytes = 0,
		.min_interval = 0,
		.max_interval = 0,
		.fixed_intervals = 
		{ 
			{ 
				.time = 20, 
				.bytes = 33
			},
			{ 
				.time = 40, 
				.bytes = 65
			},
		},
		.sampling = 8000,
		.flags = 0,
	},
#if 0
	{
		.name = "G726_16",
		.desc = "G726 (16kbps)",
		.codec_id = SNGTC_CODEC_G726_16,
		.payload = 124,
		.default_interval = 20,
		.per_interval_bytes = 20,
		.min_interval = 10,
		.max_interval = 50,
		.fixed_intervals = { { 0 , 0 } },
	},
	{
		.name = "G726_24",
		.desc = "G726 (24kbps)",
		.codec_id = SNGTC_CODEC_G726_24,
		.payload = 123,
		.default_interval = 20,
		.per_interval_bytes = 30,
		.min_interval = 10,
		.max_interval = 50,
		.fixed_intervals = { { 0 , 0 } },
	},
	{
		.name = "G726_40",
		.desc = "G726 (40kbps)",
		.codec_id = SNGTC_CODEC_G726_40,
		.payload = 121,
		.default_interval = 20,
		.per_interval_bytes = 50,
		.min_interval = 10,
		.max_interval = 50,
		.fixed_intervals = { { 0 , 0 } },
	},
#endif
	{
		.name = NULL,
		.codec_id = 0,
		.payload = 0,
		.default_interval = 0,
		.per_interval_bytes = 0,
		.min_interval = 0,
		.max_interval = 0,
		.fixed_intervals = { { 0 , 0 } },
		.sampling = 0,
		.flags = 0,
	},
};

static void stop_handler(int signum)
{
	globals.running = 0;
}

static int create_rtp_port(void *usr_priv, uint32_t host_ip, uint32_t *p_rtp_port, void **rtp_fd)
{
	*p_rtp_port = globals.base_port;
	*rtp_fd = (void *)(long)globals.base_port;
	globals.base_port += 2;
	return 0;
}

static int release_rtp_port(void *usr_priv, uint32_t host_ip, uint32_t p_rtp_port, void *rtp_fd)
{
	return 0;
}

static int create_rtp_socket(void *usr_priv, sngtc_codec_request_leg_t *codec_req, 
		             sngtc_codec_reply_leg_t *codec_reply, void **rtp_fd)
{
	RtpSession *session = NULL;
 	uint32_t local_port = (int)(long)*rtp_fd;

	session = rtp_session_new(RTP_SESSION_SENDRECV);	

	rtp_session_set_scheduling_mode(session, FALSE);

	rtp_session_set_blocking_mode(session, FALSE);

	rtp_session_set_local_addr(session, globals.bindip_str, local_port);

	rtp_session_set_connected_mode(session, TRUE);

	rtp_session_set_symmetric_rtp(session, FALSE);

	rtp_session_enable_adaptive_jitter_compensation(session, FALSE);

	/* without jitter buffer it seems we drop once in a while, we should test later again without it */
	rtp_session_enable_jitter_buffer(session, TRUE);

	rtp_session_enable_rtcp(session, FALSE);

	rtp_session_set_jitter_compensation(session, 0);

	*rtp_fd = session;
	
	local_sngtc_logger(SNGTC_LOGLEVEL_DEBUG, "Created rtp session %p with local port %d\n", session, local_port);

	return 0;	
}

static int destroy_rtp_socket(void *usr_priv, void *fd)
{
	RtpSession *session = fd;

	rtp_session_destroy(session);
	return 0;
}

static int local_sngtc_logger(int level, char *fmt, ...)
{
	struct timeval currtime;
	struct tm currtime_tm;
	time_t currsec;
	pid_t thread_id;
	char *data = NULL;
	void *thread_id_container = NULL;
	const char *levelname = NULL;

	va_list ap;

	/* increase errors if any */
	if (level == SNGTC_LOGLEVEL_WARN || level == SNGTC_LOGLEVEL_ERROR || level == SNGTC_LOGLEVEL_CRIT) {
		pthread_mutex_lock(&globals.the_mutex);
		globals.errors++;
		pthread_mutex_unlock(&globals.the_mutex);
	}

	if (level == SNGTC_LOGLEVEL_BERT && !globals.bert_silent) {
		/* BERT stuff is always logged unless -bert_silent is provided*/
		levelname = "BERT";
		goto dolog;
	}

	/* if no verbosity, do not log anything at all */
	if (globals.verbosity == VERBOSITY_NONE) {
		return 0;
	}

	/* if user requested only errors, then discard anything else */
	if (globals.verbosity == VERBOSITY_ONLY_ERRORS) {
		if (level != SNGTC_LOGLEVEL_WARN && level != SNGTC_LOGLEVEL_ERROR && level != SNGTC_LOGLEVEL_CRIT) {
			return 0;
		}
	}

	levelname = SNGTC_LOGLEVEL_DECODE(level);

dolog:

	/* log everything else ... */

	/* fancy way of getting the thread id ... I could keep calling syscall every single time but I wanted to have some fun with pthread keys ... */
	if ((thread_id_container = pthread_getspecific(globals.thread_id_key)) == NULL) {
		thread_id = syscall(SYS_gettid);
		pthread_setspecific(globals.thread_id_key, (void *)(long)thread_id);
	} else {
		thread_id = (long)thread_id_container;
	}
	
	va_start(ap, fmt);

	vasprintf(&data, fmt, ap);

	currsec = time(NULL);
	gettimeofday(&currtime, NULL);

	localtime_r(&currsec, &currtime_tm);

	fprintf(stderr, "[%02d:%02d:%02d:%03lu] [p=%d t=%d] [%s] %s\n", 
		currtime_tm.tm_hour, currtime_tm.tm_min,
		currtime_tm.tm_sec, (unsigned long)currtime.tv_usec/1000,
		globals.pid, thread_id, levelname, data);

	free(data);

	va_end(ap);

	return 0;
}

static void local_ortp_logger(OrtpLogLevel level, const char *fmt, va_list args) 
{
	struct timeval currtime;
	struct tm currtime_tm;
	time_t currsec;
	const char *ortp_level_str = NULL;
	void *thread_id_container = NULL;
	pid_t thread_id;
	char *data = NULL;

	/* increase errors if any */
	if (level == ORTP_WARNING || level == ORTP_ERROR || level == ORTP_FATAL) {
		pthread_mutex_lock(&globals.the_mutex);
		globals.errors++;
		pthread_mutex_unlock(&globals.the_mutex);
	}

	/* if no verbosity, do not log anything at all */
	if (globals.verbosity == VERBOSITY_NONE) {
		return;
	}

	/* if user requested only errors, then discard anything else */
	if (globals.verbosity == VERBOSITY_ONLY_ERRORS) {
		if (level != ORTP_WARNING && level != ORTP_ERROR && level != ORTP_FATAL) {
			return;
		}
	}

	/* fancy way of getting the thread id ... I could keep calling syscall every single time but I wanted to have some fun with pthread keys ... */
	if ((thread_id_container = pthread_getspecific(globals.thread_id_key)) == NULL) {
		thread_id = syscall(SYS_gettid);
		pthread_setspecific(globals.thread_id_key, (void *)(long)thread_id);
	} else {
		thread_id = (long)thread_id_container;
	}

	switch (level) {
	case ORTP_DEBUG:
		ortp_level_str = "ORTP_DEBUG";
		break;
	case ORTP_MESSAGE:
		ortp_level_str = "ORTP_MESSAGE";
		break;
	case ORTP_WARNING:
		ortp_level_str = "ORTP_WARNING";
		break;
	case ORTP_ERROR:
		ortp_level_str = "ORTP_ERROR";
		break;
	case ORTP_FATAL:
		ortp_level_str = "ORTP_FATAL";
		break;
	default:
		ortp_level_str = "ORTP_UNKNOWN";
		break;
	}

	vasprintf(&data, fmt, args);

	currsec = time(NULL);
	gettimeofday(&currtime, NULL);

	localtime_r(&currsec, &currtime_tm);

	fprintf(stderr, "[%02d:%02d:%02d:%03lu] [p=%d t=%d] [%s] %s\n", 
			currtime_tm.tm_hour, currtime_tm.tm_min,
			currtime_tm.tm_sec, (unsigned long)currtime.tv_usec/1000,
			globals.pid, thread_id, ortp_level_str, data);

	free(data);
}

static void print_usage(void)
{
	fprintf(stderr, 
 		"Sangoma Transcoder SOAP Client Utility\n\n"
		"sngtc_client usage:\n\n"
		"-bindip <IP address> - This is the IP address that will be used for RTP connections. (required)\n\n"
		"-incodec <codec name> - This is the input codec name. See -codecs option for a list of supported codecs. (required)\n\n"
		"-outcodec <codec name> - This is the output codec name. See -codecs option for a list of supported codecs. (required)\n\n"
		"-baseport <port>- Print a list of supported codecs and exit. If not specified default %d is used.\n\n"
		"-interval <time> - Interval (ptime) for RTP (common values are 20,30,40 ... defaults to 20). The value will be used for both input and output codecs. The default depends on the codec selected.\n\n"
		"-infile <file> - Input file to read media from. Defaults to standard input if not specified. Option mandatory if -sessions > 1 is specified.\n\n"
		"-outfile <file> - Output file to write media to. Defaults to standard output if not specified. Option mandatory if -sessions > 1 is specified. If -sessions > 1, each session appends a number to the file name.\n\n"
		"-sessions <number> - How many transcoding sessions to launch. Defaults to 1.\n\n"
		"-soapaddr <URL> - SOAP URL to contact to create transcoding sesions. If not specified the default Sangoma transcoding server URL will be used.\n\n"
		"-rtpdebug - Print RTP debugging messages to stderr.\n\n"
		"-verbosity - Application verbosity. 0 will not print anything at all, not even errors. 1 will print errors only. 2 will print everything. (default is 1)\n\n"
		"-wav - Write a wav file as output when using L16_1 as the output codec.\n\n"
		"-bert - Do bert-like testing. The -infile, -outfile, -incodec -outcodec and -wav options, and possibly others, will be ignored if this option is set).\n\n"
		"-bert_silent - No printing during BERT test\n\n"
		"-linger - Do not destroy transcoding sessions until all sessions are done streaming.\n\n"
		"-codecs - Print a list of supported codecs and exit.\n\n",
		DEFAULT_BASE_UDP_PORT);
}

static void print_codecs(void)
{
	int i;
	int j;
	for (i = 0; global_codecs[i].name; i++) {
		printf("%s - %s\n", global_codecs[i].name, global_codecs[i].desc);
		printf("\tIntervals: ");
		if (!global_codecs[i].fixed_intervals[0].time) {
			for (j = global_codecs[i].min_interval; j <= global_codecs[i].max_interval; j += INTERVAL_STEP) {
				printf("%d ", j);
			}
			printf("\n");
		} else {
			for (j = 0; global_codecs[i].fixed_intervals[j].time; j++) {
				printf("%d ", global_codecs[i].fixed_intervals[j].time);
			}
			printf("\n");
		}
		printf("\n");
	}
}

static codec_t *get_codec_from_str(const char *codec_name)
{
	int i;
	for (i = 0; global_codecs[i].name; i++) {
		if (!strcasecmp(codec_name, global_codecs[i].name)) {
			return &global_codecs[i];
		}
	}
	return NULL;
}

static int get_interval_bytes(codec_t *codec, int interval, int *bytes)
{
	int i = 0;

	*bytes = 0;

	if (!codec->fixed_intervals[0].time) {
		if (codec->max_interval < interval) {
			local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "codec interval for %s cannot be bigger than %d\n", codec->name, codec->max_interval);
			return -1;
		}
		if (codec->min_interval > interval) {
			local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "codec interval for %s cannot be smaller than %d\n", codec->name, codec->min_interval);
			return -1;
		}
		*bytes = (interval / INTERVAL_STEP) * codec->per_interval_bytes;
	} else {
		for (i = 0; codec->fixed_intervals[i].bytes; i++) {
			if (codec->fixed_intervals[i].time == interval) {
				*bytes = codec->fixed_intervals[i].bytes;
				break;
			}
		}
		if (!*bytes) {
			local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "codec interval %d did not match any of the supported intervals for codec %s, use -codecs option to see a list of supported intervals for each codec.\n", interval, codec->name);
			return -1;
		}
	}
	return 0;
}

/* check if the given file is a wave file and skip the header if it is */
#define WAVE_CHUNK_ID "RIFF"
#define WAVE_FMT "WAVEfmt "
#define WAVE_HEADER_LEN 44
static int skip_wave_header(FILE *f)
{
	char rbuff[10] = { 0 };
	unsigned int hz = 0;
	unsigned int hs = 0;
	unsigned short fmt = 0;
	unsigned short chans = 0;
	unsigned int size = 0;

	/* check chunk id */
	if (fread(rbuff, 1, 4, f) != 4) {
		local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Unable to read wav chunk id\n");
		goto error;
	}
	rbuff[4] = 0;

	if (strncasecmp(rbuff, WAVE_CHUNK_ID, sizeof(WAVE_CHUNK_ID)-1)) {
		goto notwave;
	}

	/* read chunk size */
	if (fread(&size, 1, 4, f) != 4) {
		local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Unable to read wav chunk size\n");
		goto error;
	}

	/* check format and sub chunk id */
	if (fread(rbuff, 1, 8, f) != 8) {
		local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Unable to read wav format and sub chunk id\n");
		goto error;
	}
	rbuff[8] = 0;

	if (strncasecmp(rbuff, WAVE_FMT, sizeof(WAVE_FMT)-1)) {
		goto notwave;
	}

	/* At this point we know is a wav file ... */

	/* validate sub chunk size */
	if (fread(&hs, 1, 4, f) != 4) {
		local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Unable to read wav sub chunk size\n");
		goto error;
	}
	
	if (hs != 16) {
		local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Unsupported wav sub chunk size %d\n", hs);
		goto error;
	}

	/* validate audio format */
	if (fread(&fmt, 1, 2, f) != 2) {
		local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Unable to read wav audio format\n");
		goto error;
	}
	
	if (fmt != 1) {
		local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Unsupported wav audio format %d, we only support 1 (PCM)\n", fmt);
		goto error;
	}

	/* validate channels */
	if (fread(&chans, 1, 2, f) != 2) {
		local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Unable to read wav channels\n");
		goto error;
	}
	
	if (chans != 1) {
		local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Unsupported wav channels %d, we only support 1 (mono)\n", chans);
		goto error;
	}

	/* validate sampling rate */
	if (fread(&hz, 1, 2, f) != 2) {
		local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Unable to read wav sampling rate\n");
		goto error;
	}
	
	if (hz != globals.in_codec->sampling) {
		local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Invalid input wave sampling rate %dHz, output format %s requires %dHz\n", 
				hz, globals.in_codec->name, globals.in_codec->sampling);
		goto error;
	}

	local_sngtc_logger(SNGTC_LOGLEVEL_DEBUG, "Found input PCM mono wave of %d bytes at %dHz, skipping header ...\n", size, hz);
	fseek(f, WAVE_HEADER_LEN, SEEK_SET);

	return 0;

notwave:
	fseek(f, 0 , SEEK_SET);
	return 0;

error:
	return -1;
}

#define MAX_BERT_SYNC_ERR 5
static void *run_transcoding_session(void *arg)
{
	int err;
	char ipbuff[255];
	struct pollfd pollrtp;
	struct timespec timesent;
	struct timespec timerecv; 
	struct timespec globalstart;
	struct timespec globalend; 
	uint64_t sentus = 0;
	uint64_t recvus = 0;
	uint64_t diffus = 0;
	uint64_t maxus = 0;
	uint64_t minus = 100000;
	uint64_t avgus = 0;
	uint64_t globalstartus = 0;
	uint64_t globalendus = 0;
	uint64_t globalus = 0;
	uint64_t globalusageus = 0;
	struct rusage start_usage;
	struct rusage end_usage;
	int txpackets = 0;
	int rxpackets = 0;
	int rx_samples_per_interval = 0;
	int tx_samples_per_interval = 0;
	uint32_t rx_ts = 0;
	int tx_ts = 0;
	int have_more;
	uint8_t media_buffer[4096];
	char output_filename[sizeof(globals.out_file_str)+10];
	char input_filename[sizeof(globals.out_file_str)+10];
	FILE *out_file = NULL;
	FILE *in_file = NULL;
	RtpSession *tx_rtp = NULL;
	RtpSession *rx_rtp = NULL;
	sngtc_codec_request_t codec_request;
	sngtc_codec_reply_t codec_reply;
	pthread_t thread_id;
	struct {
		uint8_t tx_inc;
		uint8_t test_data;
		uint8_t sync;
		int32_t byte_since_sync;
		int32_t sync_err;
	} bert;

	int sessindex = (int)(long)arg;
	uint8_t session_created = 0;

	/* wait here until the master threads launches all threads */
	pthread_mutex_lock(&globals.the_mutex);

	/* go! */
	globals.sessions_streaming++;

	thread_id = syscall(SYS_gettid);
	pthread_setspecific(globals.thread_id_key, (void *)(long)thread_id);
	local_sngtc_logger(SNGTC_LOGLEVEL_DEBUG, "Running session %d in thread %d\n", sessindex, thread_id);

	if (!strlen(globals.out_file_str)) {
		out_file = stdout;
	} else {
		if (globals.sessions > 1) {
			snprintf(output_filename, sizeof(output_filename), "%s.%d", globals.out_file_str, sessindex);
		} else {
			snprintf(output_filename, sizeof(output_filename), "%s", globals.out_file_str);
		}
		out_file = fopen(output_filename, "w");
		if (!out_file) {
			local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Failed to open output file %s\n", output_filename);
			pthread_mutex_unlock(&globals.the_mutex);
			goto donethread;
		}
		if (globals.write_wav) {
			/* we only support writing L16 files, but that was already validated on startup so everything here
			 * assumes L16 */
			FILE *f = out_file;

			unsigned int hz = globals.out_codec->sampling;
			unsigned int bhz = globals.out_codec->sampling * 2;
			unsigned int hs = 16;
			unsigned short fmt = 1;
			unsigned short chans = 1;
			unsigned short bysam = 2;
			unsigned short bisam = 16;
			unsigned int size = 0;

			/* write the wav header (http://ccrma.stanford.edu/courses/422/projects/WaveFormat/) */
			fseek(f, 0 , SEEK_SET);
			if (fwrite(WAVE_CHUNK_ID, 1, 4, f) != 4) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Unable to write wav header\n");
			}
			if (fwrite(&size, 1, 4, f) != 4) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Unable to write wav header\n");
			}
			if (fwrite("WAVEfmt ", 1, 8, f) != 8) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Unable to write wav header\n");
			}
			if (fwrite(&hs, 1, 4, f) != 4) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Unable to write wav header\n");
			}
			if (fwrite(&fmt, 1, 2, f) != 2) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Unable to write wav header\n");
			}
			if (fwrite(&chans, 1, 2, f) != 2) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Unable to write wav header\n");
			}
			if (fwrite(&hz, 1, 4, f) != 4) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Unable to write wav header\n");
			}
			if (fwrite(&bhz, 1, 4, f) != 4) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Unable to write wav header\n");
			}
			if (fwrite(&bysam, 1, 2, f) != 2) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Unable to write wav header\n");
			}
			if (fwrite(&bisam, 1, 2, f) != 2) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Unable to write wav header\n");
			}
			if (fwrite("data", 1, 4, f) != 4) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Unable to write wav header\n");
			}
			if (fwrite(&size, 1, 4, f) != 4) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Unable to write wav header\n");
			}
			/* done with the 44 bytes header */
		}
	}

	if (!strlen(globals.in_file_str)) {
		in_file = stdin;
	} else {
		in_file = fopen(globals.in_file_str, "r");
		if (!in_file) {
			if (globals.sessions > 1) {
				snprintf(input_filename, sizeof(input_filename), "%s.%d", globals.in_file_str, sessindex);
				in_file = fopen(input_filename, "r");
				if (!in_file) {
					local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Failed to open input file %s\n", input_filename);
					fclose(out_file);
					pthread_mutex_unlock(&globals.the_mutex);
					goto donethread;
				}
			} else {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Failed to open input file %s\n", globals.in_file_str);
				fclose(out_file);
				pthread_mutex_unlock(&globals.the_mutex);
				goto donethread;
			}
		}
		if (skip_wave_header(in_file)) {
			pthread_mutex_unlock(&globals.the_mutex);
			goto donethread;
		}
	}


	memset(&timesent, 0, sizeof(timesent));
	memset(&timerecv, 0, sizeof(timerecv));
	memset(&globalstart, 0, sizeof(globalstart));
	memset(&globalend, 0, sizeof(globalend));
	memset(&codec_request, 0, sizeof(codec_request));

	codec_request.a.host_ip = globals.bindip_addr;
	codec_request.a.codec_id = globals.in_codec->codec_id;
	codec_request.a.ms = globals.in_interval;

	codec_request.b.host_ip = globals.bindip_addr;
	codec_request.b.codec_id = globals.out_codec->codec_id;
	codec_request.b.ms = globals.out_interval;

	err = sngtc_create_transcoding_session(&codec_request, &codec_reply, 0);
	if (err) {
		local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Failed to create transcoding session\n");
		pthread_mutex_unlock(&globals.the_mutex);
		goto donethread;
	}
	session_created = 1;
	tx_rtp = codec_reply.tx_fd;
	rx_rtp = codec_reply.rx_fd;

	/* tell the RTP stack the payload to expect */
	rtp_session_set_payload_type(tx_rtp, globals.in_codec->payload);
	rtp_session_set_payload_type(rx_rtp, globals.out_codec->payload);

	/* tell the rtp stack the remote address for each RTP session */
	snprintf(ipbuff, sizeof(ipbuff), SNGTC_NIPV4_FMT, SNGTC_NIPV4(codec_reply.a.codec_ip));
	rtp_session_set_remote_addr(tx_rtp, ipbuff, codec_reply.a.codec_udp_port);
	local_sngtc_logger(SNGTC_LOGLEVEL_DEBUG, "Sending %s (%d/%d) to %s:%d in packets of %d bytes (interval = %d)\n", 
			globals.in_codec->name, codec_reply.a.iana_code, codec_reply.tx_iana, ipbuff, codec_reply.a.codec_udp_port,
			globals.in_bytes, globals.in_interval);
	
	snprintf(ipbuff, sizeof(ipbuff), SNGTC_NIPV4_FMT, SNGTC_NIPV4(codec_reply.b.codec_ip));
	rtp_session_set_remote_addr(rx_rtp, ipbuff, codec_reply.b.codec_udp_port);
	local_sngtc_logger(SNGTC_LOGLEVEL_DEBUG, "Receiving %s (%d/%d) from %s:%d in packets of %d bytes (interval = %d)\n", 
			globals.out_codec->name, codec_reply.b.iana_code, codec_reply.rx_iana, ipbuff, codec_reply.b.codec_udp_port,
			globals.out_bytes, globals.out_interval);

	/* calculate samples per interval depending on the sampling rate */
	rx_samples_per_interval = ((globals.out_interval * globals.out_codec->sampling)/MILLISECONDS_IN_SECOND);
	tx_samples_per_interval = ((globals.in_interval * globals.in_codec->sampling)/MILLISECONDS_IN_SECOND);

	getrusage(RUSAGE_SELF, &start_usage);
	clock_gettime(CLOCK_MONOTONIC, &globalstart);

	memset(&pollrtp, 0, sizeof(pollrtp));
	/* 
	 * we drive output by input, 
	 * we can add proper interval timeout to poll 
	 * if we want independent output 
	 * */
	pollrtp.fd = rtp_session_get_rtp_socket(rx_rtp);
	pollrtp.events = POLLIN | POLLERR;

	/* release lock so other threads can setup their sessions */
	pthread_mutex_unlock(&globals.the_mutex);

	/* do media reading/writing */
	rx_ts = rx_samples_per_interval;
	while (globals.running)
	{
		err = poll(&pollrtp, 1, 1000);
		if (err < 0 && errno == EINTR) {
			continue;
		}
		
		if (err < 0) {
			local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "poll failed: %s\n", strerror(errno));
			break;
		}

		if (!err) {
			local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "timed out waiting for input!\n");
			if (globals.do_bert) {
				continue;
			}
			break;
		}

		if (pollrtp.revents & POLLERR) {
			local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "POLLERR on fd %d!\n", pollrtp.fd);
			break;
		}

		if (pollrtp.revents & POLLIN) {
readagain:
			have_more = 0;
			err = rtp_session_recv_with_ts(rx_rtp, media_buffer, globals.out_bytes, rx_ts, &have_more);

			rx_ts += rx_samples_per_interval;
			if (err && err != globals.out_bytes) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Read unexpected number of bytes %d != %d from fd %d!\n", err, globals.out_bytes, pollrtp.fd);
				break;
			}

			if (!err) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Could not read from RTP, rc = %d, have_more = %d, rxpackets = %d, errno = %d\n", err, have_more, rxpackets, errno);
				break;
			}

			if (have_more) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "We should not have more!\n");
				if (globals.do_bert) {
					continue;
				}
				break;
			}

			if (timesent.tv_sec) {
				if (clock_gettime(CLOCK_MONOTONIC, &timerecv)) {
					perror("gettime");
					local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "clock_gettime failed!\n");
					break;
				}
				sentus = ((timesent.tv_sec * 1000000) + (timesent.tv_nsec / 1000));
				recvus = ((timerecv.tv_sec * 1000000) + (timerecv.tv_nsec / 1000));
				diffus = (recvus - sentus);

				if (recvus <= sentus) {
					/* Its possible to get time backward in time. Do not error out */
					local_sngtc_logger(SNGTC_LOGLEVEL_DEBUG, "recvus is %llu and sentus is %llu!\n",
									   (unsigned long long)recvus, (unsigned long long)sentus);
					
				} else {

					if (diffus > maxus) {
						maxus = diffus;	
					}
					if (diffus < minus) {
						minus = diffus;
					}
					avgus = (avgus + diffus) / 2;

				}
			}

			if (!globals.do_bert) {

				if ((globals.out_codec->flags & CODEC_FLAG_BYTESWAP)) {
					int i;
					unsigned short *dst_s = (unsigned short *)media_buffer;
					const unsigned short *src_s = (const unsigned short *)media_buffer;
					for (i = 0; i < (err/2); i++) {
						dst_s[i] = (src_s[i] << 8) | (src_s[i] >> 8);
					}
				}

				fwrite(media_buffer, 1, err, out_file);
				/* only flush when using stdout as the output, otherwise eventual delay is 
				 * introduced here causing vocallo to insert silence frames, and not flushing on stdout
				 * causes applications on the other side (ie, other instance of this program) to not
				 * get the audio on time */
				if (out_file == stdout) {
					fflush(out_file);
				}

			} else if (globals.in_codec->codec_id == SNGTC_CODEC_PCMU && 
				   globals.in_codec->codec_id == globals.out_codec->codec_id) {
				/* do BERT stuff here for PCMU */
				int i;
				for (i = 0; i < err; i++) {
					if (!bert.sync) {
						if (bert.test_data != media_buffer[i]) {
							bert.test_data = media_buffer[i];
							bert.test_data = (bert.test_data + 1) % 0x1F;
						} else {
							if (!bert.sync_err) {
								local_sngtc_logger(SNGTC_LOGLEVEL_INFO, "BERT test sync, data = %x, sample index = %d, length = %d\n",
										media_buffer[i], i, err);
								
								pthread_mutex_lock(&globals.the_mutex);
								globals.sessions_sync++;
								pthread_mutex_unlock(&globals.the_mutex);
								fflush(stdout);
							}
							bert.byte_since_sync = 1;
							bert.sync = 1;
						}
					} else {
						bert.byte_since_sync++;
						bert.test_data = (bert.test_data + 1) % 0x1F;
						if (bert.test_data != media_buffer[i]) {
							if (bert.sync_err > MAX_BERT_SYNC_ERR) {	
								local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "BERT test error, expecting = %d(%x), rx = %d(%x), sample index = %d, bss = %d serr = %d\n",
										bert.test_data, bert.test_data, media_buffer[i], media_buffer[i], i, bert.byte_since_sync,bert.sync_err);
							} else if (bert.sync_err) {
								local_sngtc_logger(SNGTC_LOGLEVEL_BERT, "BERT test warning, expecting = %d(%x), rx = %d(%x), sample index = %d, bss = %d serr=%d\n",
										bert.test_data, bert.test_data, media_buffer[i], media_buffer[i], i, bert.byte_since_sync,bert.sync_err);
							}
							bert.test_data = media_buffer[i];
							bert.test_data++;
							bert.sync_err++;
							globals.sync_err += bert.sync_err;
							bert.sync = 0;
						}
					}
				}
			}
			rxpackets++;
			globals.rxpackets++;

			/* check if there is something in the read queue still */
			err = poll(&pollrtp, 1, 0);
			if (err < 0) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Poll failed, rc = %d, %s\n", err, strerror(errno));
				break;
			}

			if (err && pollrtp.revents & POLLIN) {
				local_sngtc_logger(SNGTC_LOGLEVEL_DEBUG, "reading again!\n");
				goto readagain;
			}


			if (!globals.do_bert) {

				if (feof(in_file)) {
					local_sngtc_logger(SNGTC_LOGLEVEL_DEBUG, "End of file\n");
					break;
				}

				err = fread(media_buffer, 1, globals.in_bytes, in_file);
				if (err != globals.in_bytes) {
					if (feof(in_file)) {
						local_sngtc_logger(SNGTC_LOGLEVEL_DEBUG, "End of file\n");
						break;
					}
					if (ferror(in_file)) {
						local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Could not read %d bytes from input file\n", globals.in_bytes);
						break;
					}
					local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "fread failed but no EOF or error?\n");
					break;
				}
				if ((globals.in_codec->flags & CODEC_FLAG_BYTESWAP)) {
					int i;
					unsigned short *dst_s = (unsigned short *)media_buffer;
					const unsigned short *src_s = (const unsigned short *)media_buffer;
					for (i = 0; i < (err/2); i++) {
						dst_s[i] = (src_s[i] << 8) | (src_s[i] >> 8);
					}
				}
			} else {
				int i;
				/* fill up BERT tx here! */
				for (i = 0; i < globals.in_bytes; i++) {
					media_buffer[i] = bert.tx_inc;
					bert.tx_inc++;
					bert.tx_inc = bert.tx_inc % 0x1F;
				}
			}


			rtp_session_send_with_ts(tx_rtp, media_buffer, globals.in_bytes, tx_ts);
			tx_ts += tx_samples_per_interval;
			txpackets++;
			if (clock_gettime(CLOCK_MONOTONIC, &timesent)) {
				perror("gettime");
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "clock_gettime failed!\n");
				break;
			}
			
			if (globals.timeout) {
				clock_gettime(CLOCK_MONOTONIC, &globalend);
				globalstartus = (globalstart.tv_sec * 1000000) + (globalstart.tv_nsec / 1000);
				globalendus = (globalend.tv_sec * 1000000) + (globalend.tv_nsec / 1000);
				globalus = globalendus - globalstartus;
				if (globals.timeout < globalus) {
					local_sngtc_logger(SNGTC_LOGLEVEL_DEBUG, "Test timeout %i %i!!\n",
							globals.timeout, globalus);
					globals.running=0;
					break;
				}
			}

		} else {
			local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "No timeout but no POLLIN either!!\n");
			break;
		}

	}

	if (in_file != stdin) {
		fclose(in_file);
	}
	if (out_file != stdout) {
		if (globals.write_wav) {
			/* update header */
			off_t cur, end;
			int datalen, filelen, bytes;

			cur = ftello(out_file);
			fseek(out_file, 0, SEEK_END);
			end = ftello(out_file);
			bytes = end - WAVE_HEADER_LEN;
			datalen = bytes;
			filelen = 36 + bytes;

			fseek(out_file, 4, SEEK_SET);
			fwrite(&filelen, 1, 4, out_file);

			fseek(out_file, 40, SEEK_SET);
			fwrite(&datalen, 1, 4, out_file);

			fseek(out_file, cur, SEEK_SET);
		}
		fclose(out_file);
	}

	clock_gettime(CLOCK_MONOTONIC, &globalend);
	getrusage(RUSAGE_SELF, &end_usage);

	globalusageus = ((end_usage.ru_utime.tv_sec - start_usage.ru_utime.tv_sec) * 1000000) + end_usage.ru_utime.tv_usec - start_usage.ru_utime.tv_usec;	
	globalusageus += ((end_usage.ru_stime.tv_sec - start_usage.ru_stime.tv_sec) * 1000000) + end_usage.ru_stime.tv_usec - start_usage.ru_stime.tv_usec;

	globalstartus = (globalstart.tv_sec * 1000000) + (globalstart.tv_nsec / 1000);
	globalendus = (globalend.tv_sec * 1000000) + (globalend.tv_nsec / 1000);
	globalus = globalendus - globalstartus;

	local_sngtc_logger(SNGTC_LOGLEVEL_DEBUG, "Tx packets = %d, Rx packets = %d\n", txpackets, rxpackets);
	local_sngtc_logger(SNGTC_LOGLEVEL_DEBUG, "globalus = %llu, maxus = %llu, minus = %llu, avgus = %llu, global user usage = %llu\n", (unsigned long long)globalus, (unsigned long long)maxus, 
			(unsigned long long)minus, (unsigned long long)avgus, 
			(unsigned long long)globalusageus);
donethread:
	

	local_sngtc_logger(SNGTC_LOGLEVEL_DEBUG, "Finished streaming session %d in thread %d\n", sessindex, thread_id);

	pthread_mutex_lock(&globals.the_mutex);
	
	globals.sessions_streaming--;

	pthread_mutex_unlock(&globals.the_mutex);

	
	if (globals.do_bert) {
		while (globals.running) {
			/* Wait for test to stop */
			usleep(500000);
		}
	}
	
	/* if -linger was specified we must wait all other sessions before freeing our transcoding session */
	if (globals.linger) {
		while (globals.sessions_streaming) {
			usleep(100000);
		}
	}

	if (session_created) {
		sngtc_free_transcoding_session(&codec_reply);
	}

	pthread_mutex_lock(&globals.the_mutex);
	
	globals.sessions_running--;

	pthread_mutex_unlock(&globals.the_mutex);

	local_sngtc_logger(SNGTC_LOGLEVEL_DEBUG, "Finished session %d in thread %d\n", sessindex, thread_id);

	return NULL;
}

#define INC_ARG(arg_i)  \
		arg_i++; \
		if (arg_i >= argc) { \
			local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "No option value was given\n"); \
			exit(1); \
		}
int main(int argc, char*argv[])
{
	int arg_i = 0;
	int res = 0;
	int detected = 0;
	int activated = 0;
	int i = 0;
	pthread_t session_thread_id;
	pthread_mutexattr_t mutexattr;
	pthread_attr_t session_thread_attr;


	/* key initialization must be done before any logging */
	res = pthread_key_create(&globals.thread_id_key, NULL);
	if (res) {
		exit(1);
	}

	if (argc <= 1) {
		print_usage();
		exit(0);
	}

	memset(&g_init_cfg, 0, sizeof(g_init_cfg));
	memset(&globals, 0, sizeof(globals));

	globals.pid = getpid();
	globals.sessions = 1;
	globals.base_port = DEFAULT_BASE_UDP_PORT;
	globals.verbosity = VERBOSITY_ONLY_ERRORS;


	for (arg_i = 1; arg_i < argc; arg_i++) {
		if (!strcasecmp(argv[arg_i], "-bindip")) {
			/* build up the binary IP address */
			char ipbuf[sizeof(globals.bindip_str)];
			char *c = 0;
			char *octet_ptr = 0;
			uint8_t octet = 0;
			int shift = 24;
			INC_ARG(arg_i);
			snprintf(globals.bindip_str, sizeof(globals.bindip_str), "%s", argv[arg_i]);
			strcpy(ipbuf, globals.bindip_str);
			octet_ptr = ipbuf;
			for ( ; ; ) {
			       	c = strchr(octet_ptr, '.');
				if (c) {
					*c = 0;
					c++;
				}

				octet = atoi(octet_ptr);
				globals.bindip_addr |= (octet << shift);
				octet_ptr = c;

				if (!shift || !octet_ptr) {
					/* we reached the end */
					break;
				}

				shift -= 8;
			}
			if (shift) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Not enough dots in IP address!\n");
				exit(1);
			}
		}
		else if (!strcasecmp(argv[arg_i], "-incodec")) {
			INC_ARG(arg_i);
			globals.in_codec = get_codec_from_str(argv[arg_i]);
			if (!globals.in_codec) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Invalid input codec %s, valid codecs:\n", argv[arg_i]);
				print_codecs();
				exit(1);
			}
		}
		else if (!strcasecmp(argv[arg_i], "-outcodec")) {
			INC_ARG(arg_i);
			globals.out_codec = get_codec_from_str(argv[arg_i]);
			if (!globals.out_codec) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Invalid output codec %s, valid codecs:\n", argv[arg_i]);
				print_codecs();
				exit(1);
			}
		}
		else if (!strcasecmp(argv[arg_i], "-baseport")) {
			INC_ARG(arg_i);
			globals.base_port = atoi(argv[arg_i]);
			if (globals.base_port <= 0) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Invalid udp port %s\n", argv[arg_i]);
				exit(1);
			}
		}
		else if (!strcasecmp(argv[arg_i], "-interval")) {
			INC_ARG(arg_i);
			globals.in_interval = atoi(argv[arg_i]);
			globals.out_interval = atoi(argv[arg_i]);
			if ((globals.in_interval % INTERVAL_STEP)) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Invalid interval %s, must be multiple of %d\n", argv[arg_i], INTERVAL_STEP);
				exit(1);
			}
		}
		else if (!strcasecmp(argv[arg_i], "-infile")) {
			INC_ARG(arg_i);
			strcpy(globals.in_file_str, argv[arg_i]);
			globals.in_file_str[sizeof(globals.in_file_str) - 1] = 0;
			if (!strlen(globals.in_file_str)) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Invalid input file string %s\n", argv[arg_i]);
				exit(1);
			}
		}
		else if (!strcasecmp(argv[arg_i], "-outfile")) {
			INC_ARG(arg_i);
			strcpy(globals.out_file_str, argv[arg_i]);
			globals.out_file_str[sizeof(globals.out_file_str) - 1] = 0;
			if (!strlen(globals.out_file_str)) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Invalid output file string %s\n", argv[arg_i]);
				exit(1);
			}
		}
		else if (!strcasecmp(argv[arg_i], "-rtpdebug")) {
			globals.rtpdebug = 1;
		}
		else if (!strcasecmp(argv[arg_i], "-verbosity")) {
			INC_ARG(arg_i);
			globals.verbosity = atoi(argv[arg_i]);
			if (globals.verbosity != VERBOSITY_ONLY_ERRORS &&
			    globals.verbosity != VERBOSITY_NONE &&
			    globals.verbosity != VERBOSITY_ALL) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Invalid verbosity level: %s\n", argv[arg_i]);
				exit(1);
			}
		}
		else if (!strcasecmp(argv[arg_i], "-sessions")) {
			INC_ARG(arg_i);
			globals.sessions = atoi(argv[arg_i]);
			if (globals.sessions < 1) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Invalid number of sessions: %s\n", argv[arg_i]);
				exit(1);
			}
		}
		else if (!strcasecmp(argv[arg_i], "-soapaddr")) {
			INC_ARG(arg_i);
			strcpy(globals.soap_addr_str, argv[arg_i]);
			globals.soap_addr_str[sizeof(globals.soap_addr_str) - 1] = 0;
			if (!strlen(globals.soap_addr_str)) {
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Invalid SOAP address URL %s\n", argv[arg_i]);
				exit(1);
			}
		}
		else if (!strcasecmp(argv[arg_i], "-codecs")) {
			print_codecs();
			exit(0);
		}
		else if (!strcasecmp(argv[arg_i], "-wav")) {
			globals.write_wav = 1;
		}
		else if (!strcasecmp(argv[arg_i], "-timeout")) {
			INC_ARG(arg_i);
			globals.timeout = atoi(argv[arg_i])*1000000;
		}
		else if (!strcasecmp(argv[arg_i], "-linger")) {
			globals.linger = 1;
		}
		else if (!strcasecmp(argv[arg_i], "-bert")) {
			globals.do_bert = 1;
			globals.in_codec = get_codec_from_str("PCMU");
			globals.out_codec = globals.in_codec;
		}
		else if (!strcasecmp(argv[arg_i], "-bert_silent")) {
			globals.bert_silent = 1;
		}
		else if (!strcasecmp(argv[arg_i], "-help")) {
			print_usage();
			exit(0);
		} else {
			local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Invalid option %s\n", argv[arg_i]);
			exit(1);
		}
	}

	/* check that we have the required information to continue ... */
	if (!globals.bindip_addr) {
		local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "-bindip option is required\n");
		exit(1);
	}

	if (globals.do_bert) {
		globals.write_wav = 0;
		if (strlen(globals.in_file_str)) {
			local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "-infile option cannot be used with -bert\n");
			exit(1);
		}
		if (strlen(globals.out_file_str)) {
			local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "-outfile option cannot be used with -bert\n");
			exit(1);
		}
	}

	if (!globals.in_codec) {
		local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "-incodec option is required\n");
		exit(1);
	}

	if (!globals.out_codec) {
		local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "-outcodec option is required\n");
		exit(1);
	}

	if (globals.write_wav && 
	    globals.out_codec->codec_id != SNGTC_CODEC_L16_1 &&
	    globals.out_codec->codec_id != SNGTC_CODEC_L16_2) {
		local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "-wav option is only valid for L16_1 and L16_2 output codec\n");
		exit(1);
	}

	/* we cannot use stdin/stdout as files if -sessions is bigger than one, otherwise it would be a mess */
	if (!globals.do_bert && (!strlen(globals.in_file_str) || 
	     !strlen(globals.out_file_str)) && globals.sessions > 1) {
		local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "-infile and -outfile options are required when launching more than one session (-session argument bigger than one)\n");
		exit(1);
	}

	/* if not specified, use default interval for the input codec */
	if (!globals.in_interval) {
		globals.in_interval = globals.in_codec->default_interval;
		globals.out_interval = globals.in_codec->default_interval;
	}

	/* verify the intervals are valid for the chosen codecs */
	if (get_interval_bytes(globals.in_codec, globals.in_interval, &globals.in_bytes)) {
		exit(1);
	}
	if (get_interval_bytes(globals.out_codec, globals.out_interval, &globals.out_bytes)) {
		exit(1);
	}

	signal(SIGINT, stop_handler);
	signal(SIGPIPE, stop_handler);

	g_init_cfg.log = local_sngtc_logger;
	g_init_cfg.create_rtp = create_rtp_socket;
	g_init_cfg.create_rtp_port = create_rtp_port;
	g_init_cfg.destroy_rtp = destroy_rtp_socket;
	g_init_cfg.release_rtp_port = release_rtp_port;

	sngtc_deactivate_modules();

	sngtc_detect_init_modules(&g_init_cfg, &detected);

	sngtc_activate_modules(&g_init_cfg, &activated);
       
#ifndef SNGTC_LIB_BIND
	if (strlen(globals.soap_addr_str)) {
		sngtc_set_soap_server_url(globals.soap_addr_str);
		local_sngtc_logger(SNGTC_LOGLEVEL_DEBUG, "Using configured SOAP server %s\n", globals.soap_addr_str);
	}
#endif

	ortp_set_log_handler(local_ortp_logger);

	ortp_init();

	/*ortp_scheduler_init();*/

	if (globals.rtpdebug) {
		ortp_set_log_level_mask(ORTP_DEBUG|ORTP_MESSAGE|ORTP_WARNING|ORTP_ERROR|ORTP_FATAL);
	} else {
		ortp_set_log_level_mask(ORTP_WARNING|ORTP_ERROR|ORTP_FATAL);
	}


	pthread_mutexattr_init(&mutexattr);
	pthread_mutexattr_settype(&mutexattr, PTHREAD_MUTEX_RECURSIVE);
	pthread_mutex_init(&globals.the_mutex, &mutexattr);
	globals.running = 1;

	if (globals.sessions > 1) {
		for (i = 1; (i <= globals.sessions && globals.running); i++) {
			pthread_attr_init(&session_thread_attr);
			pthread_attr_setdetachstate(&session_thread_attr, PTHREAD_CREATE_DETACHED);
			/* 32 bit systems can die if we leave default stack size for more than 300 sessions */
			pthread_attr_setstacksize(&session_thread_attr, 256 * 1024);
			pthread_mutex_lock(&globals.the_mutex);
			globals.sessions_running++;
			pthread_mutex_unlock(&globals.the_mutex);
			res = pthread_create(&session_thread_id, &session_thread_attr, run_transcoding_session, (void *)(long)i);
			if (res) {
				pthread_mutex_lock(&globals.the_mutex);
				globals.sessions_running--;
				pthread_mutex_unlock(&globals.the_mutex);
				local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Failed to create session thread %d: %s\n", i, strerror(errno));
				globals.running = 0;
				goto done;
			}
		}
	} else {
		local_sngtc_logger(SNGTC_LOGLEVEL_DEBUG, "Not launching any threads since only one session was requested\n");
		globals.sessions_running++;
		run_transcoding_session((void *)(long)1);
	}


done:

	/* main thread just busy waiting for the session to terminate */
	while (globals.sessions_running) {
		if (globals.do_bert) {
			local_sngtc_logger(SNGTC_LOGLEVEL_BERT, "BERT got sync sessions %04i packets %05i err %05i streaming %i/%i\r",
					   globals.sessions_sync,globals.rxpackets,globals.sync_err, globals.sessions_streaming,globals.sessions_running);
			fflush(stdout);
		}
		sleep(1);
	}

	ortp_exit();
	
	ortp_global_stats_display();

	/* this sleep is meant to workaround an inherent race for not using joinable threads (threads may not be done when exiting the process), but I can live with that */
	usleep(100000);
	
	if (globals.errors) {
		local_sngtc_logger(SNGTC_LOGLEVEL_ERROR, "Errors: %d\n", globals.errors);
		exit(1);
	}

	exit(0);
}

