/*
 * libsngtc-node
 * Sangoma Transcoder Node Library
 *
 * Copyright (C) 2010 Sangoma Technologies
 * Nenad Corbic <ncorbic@sangoma.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *
 * Moises Silva <moy@sangoma.com>
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "soapH.h"
#include "ns1.nsmap"
#include "sngtc_node.h"

sngtc_log_func_t sngtc_log_func;

static struct soap soap; // the gSOAP runtime context
sngtc_init_cfg_t g_init;
static int detected_modules=0;
static sngtc_mutex_t	sngtc_node_lock;
static char sngtc_server_url[255] = "http://127.10.10.1:64055";
int sngtc_rtp_session_table_idx=0;
sngtc_rtp_session_t sngtc_rtp_session_table[MAX_SNGTC_SESSIONS+1];

int __sngtc_free_transcoding_session (struct sngtc_codec_reply *codec_reply);

int sngtc_find_rtp_session(struct sngtc_codec_reply *codec_reply, sngtc_rtp_session_t **rtp_session)
{
	int i;
	sngtc_rtp_session_t *session;
	SNGTC_DEBUG("%s\n",__FUNCTION__);
	// lock
	for (i=0;i<=MAX_SNGTC_SESSIONS;i++) {
		session=&sngtc_rtp_session_table[i];
		if (session && session->init != 0) {
			if (session->server_rtp_index == codec_reply->codec_rtp_session_idx) {
				SNGTC_DEBUG("%s: Found rtp session 0x%08X init=%i\n", __FUNCTION__, codec_reply->codec_rtp_session_idx, session->init);
				*rtp_session=session;
				return 0;
			}
		}
	}
	
	// unlock
	return -1;
}

int sngtc_release_rtp_session(sngtc_rtp_session_t *session)
{
	SNGTC_DEBUG("%s\n",__FUNCTION__);
	if (session->init == 2) {
		session->init++;
		return 0;

	} else if (session->init == 1) {
		session->init--;
		return 0;
	} else if (session->init == 3) {
		SNGTC_DEBUG("Done with rtp session 0x%08X\n", session->server_rtp_index);
		session->init = 0;
		return 0;
	} else if (session->init == 0) {
		return 0;
	} else {
		SNGTC_ERROR("%s: Error invalid session init value %i\n",
				__FUNCTION__,session->init);
		return -1;
	}

	return -1;
}

int sngtc_get_free_rtp_session(sngtc_rtp_session_t **rtp_session)
{
	int i;
	sngtc_rtp_session_t *session;
	
	SNGTC_DEBUG("%s\n",__FUNCTION__);

	// lock
	for (i=0;i<=MAX_SNGTC_SESSIONS;i++) {
		sngtc_rtp_session_table_idx++;
		if (sngtc_rtp_session_table_idx > MAX_SNGTC_SESSIONS) {
			sngtc_rtp_session_table_idx=1;
		}

		session=&sngtc_rtp_session_table[sngtc_rtp_session_table_idx];
		if (session && session->init == 0) {
			session->init=1;
			session->index=sngtc_rtp_session_table_idx;
			*rtp_session=session;
			return 0;
		}
	}
	
	// unlock
	return -1;
}

int sngtc_get_existing_rtp_session(struct sngtc_codec_request *codec_req,  sngtc_rtp_session_t **rtp_session)
{
	int i;
	sngtc_rtp_session_t *session;
	
	SNGTC_DEBUG("%s\n",__FUNCTION__);

	// lock
	for (i=0;i<=MAX_SNGTC_SESSIONS;i++) {

		session = &sngtc_rtp_session_table[i];
		if (session && session->init == 1) {

			if (codec_req->a.codec_id == session->codec_req.b.codec_id &&
				codec_req->b.codec_id == session->codec_req.a.codec_id &&
				codec_req->a.ms == session->codec_req.b.ms &&
				codec_req->b.ms == session->codec_req.a.ms) {
	
				SNGTC_DEBUG("GOT EXISING SESSION !\n");
	
				session->init++;
				*rtp_session=session;
				return 0;
			}
		}
	}
	
	// unlock
	return -1;
}

int sngtc_detect_init_modules(sngtc_init_cfg_t *cfg, int *detected)
{
	int err;

	soap_init2(&soap, SOAP_IO_KEEPALIVE, SOAP_IO_KEEPALIVE);
	soap.max_keep_alive = -1; // at most 100 calls per keep-alive session
	
	memcpy(&g_init,cfg,sizeof(g_init));
	sngtc_log_func = g_init.log;
	detected_modules = *detected = 1;

	sngtc_mutex_init(&sngtc_node_lock,NULL);

	err = sngtc_verify_init_cfg(&g_init);
	if (err) {
		return err;
	}


	return 0;
}

int sngtc_activate_modules(sngtc_init_cfg_t *cfg, int *activated)
{
	*activated=1;
	return 0;	
}
		
int sngtc_detected_vocallo_modules(void)
{
	return 	detected_modules;	
}

		
int sngtc_deactivate_modules(void)
{
	soap_destroy(&soap); // delete deserialized class instances (for C++)
	soap_end(&soap); // remove deserialized data and clean ups
	soap_done(&soap); // detach the gSOAP context

	sngtc_mutex_destroy(&sngtc_node_lock);
	return 0;
}

int sngtc_create_transcoding_session (struct sngtc_codec_request *u_codec_req, 
			  					      struct sngtc_codec_reply *u_codec_reply,
									  int start_module)
{
	struct ns1__sngtc_codec_request ns_codec_req;
	struct ns1__sngtc_codec_reply ns_codec_reply;
	int err=-1;
	sngtc_rtp_session_t *session = NULL;

	sngtc_lock(&sngtc_node_lock);

	if (sizeof(ns_codec_reply) != sizeof(*u_codec_reply)) {
		SNGTC_ERROR("%s: Internal error ns_codec_reply=%i != codec_reply=%i\n",
					__FUNCTION__,sizeof(ns_codec_reply),sizeof(*u_codec_reply));
		goto sngtc_create_transcoding_session_exit;
	}
	
	if (sizeof(ns_codec_req) != sizeof(*u_codec_req)) {
		SNGTC_ERROR("%s: Internal error ns_codec_req=%i != codec_req=%i\n",
					__FUNCTION__,sizeof(ns_codec_req),sizeof(*u_codec_req));
		goto sngtc_create_transcoding_session_exit;
	}

	

	err = sngtc_get_existing_rtp_session(u_codec_req, &session);
	if (err) {
		err = sngtc_get_free_rtp_session(&session);
		if (err) {
			goto sngtc_create_transcoding_session_exit;
		}
	} else {
		/* We have determined that we already have an existing session we can reuse */
		memcpy(u_codec_reply, &session->codec_reply, sizeof(session->codec_reply));
		memcpy(&u_codec_reply->a, &session->codec_reply.b, sizeof(session->codec_reply.a));
		memcpy(&u_codec_reply->b, &session->codec_reply.a, sizeof(session->codec_reply.b));
	
		u_codec_reply->tx_fd = session->codec_reply.rx_fd;
		u_codec_reply->rx_fd = session->codec_reply.tx_fd;
		
		goto sngtc_create_transcoding_session_exit;
	}

	memcpy(&session->codec_req, u_codec_req, sizeof(*u_codec_req));

	/* From this point on we are requesting a new session */

	err = g_init.create_rtp_port(session->codec_req.usr_priv,
						       session->codec_req.a.host_ip,
						       &session->codec_req.a.host_udp_port,
						       &session->codec_reply.tx_fd);
	if (err) {
		goto sngtc_create_transcoding_session_exit;
	}
	session->codec_reply.a.host_udp_port	= session->codec_req.a.host_udp_port;

	err = g_init.create_rtp_port(session->codec_req.usr_priv,
						       session->codec_req.b.host_ip,
						       &session->codec_req.b.host_udp_port,
						       &session->codec_reply.rx_fd);
	if (err) {
		goto sngtc_create_transcoding_session_exit;
	}
	session->codec_reply.b.host_udp_port	= session->codec_req.b.host_udp_port;


	memcpy(&ns_codec_req, &session->codec_req, sizeof(session->codec_req));
	memcpy(&ns_codec_reply, &session->codec_reply, sizeof(session->codec_reply));
	
	
	err = soap_call_ns1__sngtc_create_transcoding_session(&soap,
												sngtc_server_url,
												NULL,
												&ns_codec_req,
												&ns_codec_reply);
	
	if (err == SOAP_OK && ns_codec_reply.result == SNTC_OK) {

		SNGTC_DEBUG("%s: Created session RTP ID = 0x%08X\n", __FUNCTION__, ns_codec_reply.codec_rtp_session_idx);
		memcpy(&session->codec_reply, &ns_codec_reply, sizeof(session->codec_reply));
		session->server_rtp_index = ns_codec_reply.codec_rtp_session_idx;
		session->server_mod_index = ns_codec_reply.codec_module_session_idx;

		err = g_init.create_rtp(session->codec_req.usr_priv,
						       &session->codec_req.a,
						       &session->codec_reply.a,
						       &session->codec_reply.tx_fd);
		if (err) {
			__sngtc_free_transcoding_session(&session->codec_reply);
			goto sngtc_create_transcoding_session_exit;
		}


		err = g_init.create_rtp(session->codec_req.usr_priv,
						       &session->codec_req.b,
						       &session->codec_reply.b,
						       &session->codec_reply.rx_fd);
		if (err) {
			__sngtc_free_transcoding_session(&session->codec_reply);
			goto sngtc_create_transcoding_session_exit;
		}

		memcpy(u_codec_req, &session->codec_req, sizeof(*u_codec_req));
		memcpy(u_codec_reply, &session->codec_reply, sizeof(*u_codec_reply));
		err = 0;
		

	} else { // an error occurred
		SNGTC_ERROR("Create Transcoding Session Error\n");
		if (err != SOAP_OK) {
			soap_print_fault(&soap, stderr); // display the SOAP fault message on the stderr stream
			err = -1;
		} else {
			err = ns_codec_reply.result;
		}
		
		goto sngtc_create_transcoding_session_exit;
	}


sngtc_create_transcoding_session_exit:

	if (err && session) {

		if (session->codec_reply.a.host_udp_port) {
			g_init.release_rtp_port(session->codec_req.usr_priv,
					       session->codec_req.a.host_ip,
					       session->codec_req.a.host_udp_port,
					       session->codec_reply.tx_fd);
		}

		if (session->codec_reply.b.host_udp_port) {
			g_init.release_rtp_port(session->codec_req.usr_priv,
					       session->codec_req.b.host_ip,
					       session->codec_req.b.host_udp_port,
					       session->codec_reply.rx_fd);
		}

		sngtc_release_rtp_session(session);
	}
	
	/* we must do this or we'll have leaks */		
	soap_destroy(&soap); // delete deserialized class instances (for C++)
	soap_end(&soap); // remove deserialized data and clean ups

	sngtc_unlock(&sngtc_node_lock);

	return err;
}

int __sngtc_free_transcoding_session (struct sngtc_codec_reply *codec_reply)
{
	struct ns1__sngtc_codec_reply ns_codec_reply;
	struct ns1__sngtc_codec_reply ns_codec_reply1;
	sngtc_rtp_session_t *session=NULL;
	int result=-1;
	int err;

	if (sizeof(ns_codec_reply) != sizeof(*codec_reply)) {
		SNGTC_ERROR("%s: Internal error ns_codec_reply=%i != codec_reply=%i\n",
					__FUNCTION__,sizeof(ns_codec_reply),sizeof(*codec_reply));
		return -1;
	}

	err=sngtc_find_rtp_session(codec_reply,&session);
	if (err) {
		return err;
	}

	if (session->init == 1 || session->init == 3) {

		memcpy(&ns_codec_reply, codec_reply, sizeof(*codec_reply));
		memcpy(&ns_codec_reply1, codec_reply, sizeof(*codec_reply));
		
		SNGTC_DEBUG("%s: Freeing session RTP ID = 0x%08X\n", __FUNCTION__, ns_codec_reply.codec_rtp_session_idx);

		err=soap_call_ns1__sngtc_free_transcoding_session(&soap,
				sngtc_server_url,
				NULL,
				&ns_codec_reply,&result);
		
		if (err == SOAP_OK && ns_codec_reply.result == SNTC_OK) {
			SNGTC_DEBUG("%s: Freed session RTP ID = 0x%08X OK\n", __FUNCTION__, ns_codec_reply.codec_rtp_session_idx);

			err=result;

			g_init.destroy_rtp(session->codec_req.usr_priv, session->codec_reply.tx_fd);
			g_init.destroy_rtp(session->codec_req.usr_priv, session->codec_reply.rx_fd);

		} else {
			SNGTC_ERROR("%s: Error freeing session RTP ID = 0x%08X Soap=%i Result=%i\n", __FUNCTION__, ns_codec_reply.codec_rtp_session_idx, err, ns_codec_reply.result);
			if (err != SOAP_OK) {
				soap_print_fault(&soap, stderr);
				err=-1;
			} else {
				err=ns_codec_reply.result;
			}
		}
	}

	sngtc_release_rtp_session(session);

	soap_destroy(&soap);
	soap_end(&soap);
	
	return err;	

}

int sngtc_free_transcoding_session (struct sngtc_codec_reply *codec_reply)
{
	int err;

	sngtc_lock(&sngtc_node_lock);
	err=__sngtc_free_transcoding_session(codec_reply);
	sngtc_unlock(&sngtc_node_lock);
	
	return err;	
}

int sngtc_set_soap_server_url(const char *url)
{
	sngtc_lock(&sngtc_node_lock);
	strncpy(sngtc_server_url, url, sizeof(sngtc_server_url));
	sngtc_server_url[sizeof(sngtc_server_url)-1] = '\0';
	sngtc_unlock(&sngtc_node_lock);
	return 0;
}

int sngtc_get_soap_server_url(char *url, size_t len)
{
	sngtc_lock(&sngtc_node_lock);
	strncpy(url, sngtc_server_url, len);
	url[len] = '\0';
	sngtc_unlock(&sngtc_node_lock);
	return 0;
}

