/*
 * Asterisk -- An open source telephony toolkit.
 *
 * Copyright (C) 2010, Sangoma Technologies Corp.
 *
 * Moises Silva <moy@sangoma.com>
 *
 * See http://www.asterisk.org for more information about
 * the Asterisk project. Please do not directly contact
 * any of the maintainers of this project for assistance;
 * the project provides a web site, mailing lists and IRC
 * channels for your use.
 *
 * This program is free software, distributed under the terms of
 * the GNU General Public License Version 2. See the LICENSE file
 * at the top of the source tree.
 */

/*! \file
 *
 * \brief codec_sangoma.c - Sangoma Vocallo codec module
 * 
 * \ingroup codecs
 */

/*** MODULEINFO
	<depend>sangoma_transcode</depend>
 ***/

/*
 * some notes about Asterisk versions and its interfaces, versions may vary slightly, I tested
 * with latest of each release, feel free to adjust if you find something inaccurate
 * > 1602 (at this time trunk) defines new RTP pluggable architecture and some new config keywords
 * >= 10601 defines data as a union instead of a pointer inside an ast_frame (f.data.ptr)
 * >= 10600 defines new CLI macros and prototypes for the callbacks 
 * the asteris/version.h macro ASTERISK_VERSION_NUM is used to keep this source compatible with all versions
 * >= 10400 asterisk.h was introduced
 *    ast_frame now has flags member
 *    the transcoding interface changed a lot, mostly the way the private structure is allocated and used
 * */
#ifdef NEW_VERSION_INCLUDE
/* Starting with Asterisk 11, they decided to remove ASTERISK_VERSION_NUM completely, so what are third party module devs supposed to do?
 * now ast_version.h simply defines a C API to retrieve the version number, which means you cannot detect the version without having to
 * compile and load a module into asterisk ... for now, hard-code the version to Asterisk 11, we need to come up with a more extensible
 * way of detecting the version moving forward. We could for example modify our Makefile to execute asterisk -V and parse the output 
 * then pass some defines here. I'd love a way to use the information obtained by objdump -t /usr/sbin/asterisk | grep asterisk_version_num
 * to somehow get the actual value of that symbol
 * */
/*#include "asterisk/ast_version.h"*/
#define ASTERISK_VERSION_NUM 110000
#else
#include "asterisk/version.h"
#endif

#if ASTERISK_VERSION_NUM >= 10400
#include "asterisk.h"
#endif

/* common Asterisk includes */
#include "asterisk/module.h"
#include "asterisk/config.h"
#include "asterisk/translate.h"
#include "asterisk/alaw.h"
#include "asterisk/utils.h"
#include "asterisk/sched.h"
#include "asterisk/cli.h"
#include "asterisk/acl.h"
#include "asterisk/app.h"

#if ASTERISK_VERSION_NUM > 10602
#include "asterisk/rtp_engine.h"
#else
#include "asterisk/rtp.h"
#endif

/* Sangoma transcoding library */
#ifdef SNGTC_DEBUG_HARD_LINK
#include <sng_tc/sng_tc.h>
#else
#include <sng_tc/sngtc_node.h>
#endif

#include <sys/ioctl.h>
#include <net/if.h>
#include <arpa/inet.h>
#include <errno.h>

AST_LIST_HEAD_NOLOCK(frame_list, ast_frame);

/* Work-around for bug 17092 fixed in 1.6.2 - http://issues.asterisk.org/view.php?id=17092 */
#if ASTERISK_VERSION_NUM < 10602
static int g_attempt_unload_conflicting_translators = 1;
#else
static int g_attempt_unload_conflicting_translators = 0;
#endif

/* we've seen delays, crashes and audio drops related to non-DAHDI timing, do not allow other timings by default */
static int g_allow_nondahdi_timing = 0;
static int g_dahdi_timing = 0;
static int g_rtpip = 0;

/* Asterisk 10 changed AST_FORMAT_XX definitions in a non-backwards compatible way, offering
   an API (ast_format_from_old_bitfield()) to perform conversion, but we still need the original
   bit definitions, therefore we define our own */
#define COMPAT_AST_FORMAT_G723_1    (1ULL << 0)
#define COMPAT_AST_FORMAT_GSM       (1ULL << 1)
#define COMPAT_AST_FORMAT_ULAW      (1ULL << 2)
#define COMPAT_AST_FORMAT_ALAW      (1ULL << 3)
#define COMPAT_AST_FORMAT_SLINEAR   (1ULL << 6)
#define COMPAT_AST_FORMAT_G729A     (1ULL << 8)
#define COMPAT_AST_FORMAT_ILBC      (1ULL << 10)
#define COMPAT_AST_FORMAT_G726      (1ULL << 11)
#define COMPAT_AST_FORMAT_G722      (1ULL << 12)
#define COMPAT_AST_FORMAT_SIREN7    (1ULL << 13)
#define COMPAT_AST_FORMAT_SLINEAR16 (1ULL << 15)

#if ASTERISK_VERSION_NUM < 100000
#define sangoma_format_id int
#define compat_format_id int
#else
#define sangoma_format_id uint64_t
#define compat_format_id enum ast_format_id
#endif

/* supported codecs */
typedef struct vocallo_codec_s {
	int codec_id;
	sangoma_format_id astformat;
	/* in versious previous to Asterisk 10 compat_format is the same as astformat, but we keep it anyways 
	   to avoid ifdef'ing all declarations */
	compat_format_id compat_format;
	int maxms;
} vocallo_codec_t;

#define sng_array_len(_a) (sizeof(_a)/sizeof(_a[0]))
vocallo_codec_t g_codec_map[] =
{
	{ SNGTC_CODEC_PCMU,      COMPAT_AST_FORMAT_ULAW,      AST_FORMAT_ULAW, 20 },
	{ SNGTC_CODEC_PCMA,      COMPAT_AST_FORMAT_ALAW,      AST_FORMAT_ALAW, 20 },
	{ SNGTC_CODEC_L16_1,     COMPAT_AST_FORMAT_SLINEAR,   AST_FORMAT_SLINEAR, 20 },
	{ SNGTC_CODEC_G729AB,    COMPAT_AST_FORMAT_G729A,     AST_FORMAT_G729A, 20 },
	{ SNGTC_CODEC_G722,      COMPAT_AST_FORMAT_G722,      AST_FORMAT_G722, 20 },
	{ SNGTC_CODEC_G726_32,   COMPAT_AST_FORMAT_G726,      AST_FORMAT_G726, 20 },
	{ SNGTC_CODEC_GSM_FR,    COMPAT_AST_FORMAT_GSM,       AST_FORMAT_GSM, 20 },
	{ SNGTC_CODEC_ILBC_133,  COMPAT_AST_FORMAT_ILBC,      AST_FORMAT_ILBC, 30 },
	{ SNGTC_CODEC_G723_1_53, COMPAT_AST_FORMAT_G723_1,    AST_FORMAT_G723_1, 30 },
#if ASTERISK_VERSION_NUM >= 10602
	{ SNGTC_CODEC_L16_2,     COMPAT_AST_FORMAT_SLINEAR16, AST_FORMAT_SLINEAR16, 20 },
	{ SNGTC_CODEC_SIREN7_32, COMPAT_AST_FORMAT_SIREN7,    AST_FORMAT_SIREN7, 20 },
#endif
};

#define MAX_CODECS_LIST 64
#define MAX_CODEC_STR 32
static unsigned int g_codec_register_list_count = 0;
static unsigned int g_codec_noregister_list_count = 0;
static char g_codec_register_list[MAX_CODECS_LIST][MAX_CODEC_STR];
static char g_codec_noregister_list[MAX_CODECS_LIST][MAX_CODEC_STR];
static char g_str_soap_addr[255] = "";

#define SANGOMA_TRANSLATOR_BUFFER_SIZE 8000
#define FAKE_FRAME_IDLE 0
#define FAKE_FRAME_REQUESTED 1
#define FAKE_FRAME_PROVIDED 2
#define FAKE_FRAME_TERMINATED 3 

#define SANGOMA_DEFAULT_SAMPLES 80*3
#define SANGOMA_DEFAULT_RTP_STACK  "asterisk" 
#define SANGOMA_DEFAULT_UDP_PORT 15000
#define SANGOMA_TRANSCODE_CONFIG "sangoma_codec.conf" 

#define SANG_SESS_WARN_TX(...) if (sess->txwarn > 10) { \
					sess->txwarn = sess->txwarn > 1000 ? 0 : sess->txwarn; \
					sess->txwarn++; \
				} else { \
					ast_log(LOG_WARNING, __VA_ARGS__); \
					sess->txwarn++; \
				}

#define SANG_SESS_WARN_RX(...) if (sess->rxwarn > 10) {  \
					sess->rxwarn = sess->rxwarn > 1000 ? 0 : sess->rxwarn; 		\
					sess->rxwarn++; \
				} else { \
					ast_log(LOG_WARNING, __VA_ARGS__); \
					sess->rxwarn++; \
				}

/*! transcoder init configuration */
static sngtc_init_cfg_t g_init_cfg;

enum sangoma_rtp_flags {
	SNG_RTP_TX = (1 << 0),
	SNG_RTP_RX = (1 << 1),
};

struct sangoma_transcoding_session;
struct sangoma_rtp_handle {
	void *ast_rtp; /* The asterisk RTP handle */
	unsigned int flags;
	ast_mutex_t rxlock;
	ast_mutex_t txlock;
	struct sangoma_transcoding_session *txsess;
	struct sangoma_transcoding_session *rxsess;
};

#if ASTERISK_VERSION_NUM >= 10800
#define subclass_integer subclass.integer
#else
#define subclass_integer subclass
#endif

/* \brief sangoma transcoding session private structure */
#if ASTERISK_VERSION_NUM >= 10400
#define asterisk_transcoding_pvt ast_trans_pvt
#define sangoma_transcoding_session sangoma_transcoding_session
#define sangoma_get_pvt(pvt) (pvt)->pvt
#else
#define asterisk_transcoding_pvt ast_translator_pvt
#define sangoma_transcoding_session ast_translator_pvt
#define sangoma_get_pvt(pvt) (pvt)
#endif
struct sangoma_transcoding_session {
	struct asterisk_transcoding_pvt *owner;
	int fakestate;
	sngtc_codec_request_t codec_request;
	sngtc_codec_reply_t codec_reply;
#ifdef SNGTC_DEBUG_HARD_LINK
	sngtc_rtp_stats_t rtp_stats;
#endif

	/* tx and rx rtp streams */
	struct sangoma_rtp_handle *txrtp;
	struct sangoma_rtp_handle *rxrtp;

	/* unique session id */
	unsigned long long sessid;
	
	/* tx calls and rx calls counters */
	unsigned long long txcount;
	unsigned long long rxcount;
	unsigned long long rxcallcount;
	unsigned long long rxnullcount;

	unsigned short txavgtime;
	unsigned short rxavgtime;
	unsigned short rxcallavgtime;

	struct timeval lasttxtime;
	struct timeval lastrxtime;
	struct timeval lastrxcalltime;

	/* frame sizes */
	int txfrsize;
	int rxfrsize;

	/* max number of queued frames in the RTP stack so far */
	int maxudpqueued;

	/* session recording */
	FILE *rxfile;
	FILE *txfile;
	struct ast_smoother *rxsmoother;

	/* seq counters */
  	int lastrxseqno;
	long lostrxpackets;
	long ignoredrxpackets;

	/* warning counters */
	unsigned short rxwarn;
	unsigned short txwarn;
#if ASTERISK_VERSION_NUM < 10400
	/* this duplicates some info like sangoma_transcoding_session which is the same as owner 
	 * but keeps the processing code across asterisk versions cleaner (less if ASTERISK_VERSION stuff) */
	int samples;
	int srcfmt;
	int dstfmt;
	struct sangoma_transcoding_session *t;
	struct ast_frame f;
#endif
	AST_LIST_ENTRY(sangoma_transcoding_session) entry;
};

/* \brief simple container to list all registered asterisk translators in this sangoma module */
struct sangoma_translator {
	struct ast_translator t;
	AST_LIST_ENTRY(sangoma_translator) entry;
};

/* \brief global scheduling context */
#if ASTERISK_VERSION_NUM < 100000
static struct sched_context *g_sched = NULL;
#else
static struct ast_sched_context *g_sched = NULL;
#define sched_context_destroy ast_sched_context_destroy
#define sched_context_create ast_sched_context_create
#endif

/* \brief global session id, the list of sangoma translator sessions lock protects the global next id as well */
static unsigned long long g_next_session_id = 0;
static unsigned long long g_total_sessions  = 0;

/* \brief list of sangoma translators, populated on module loading depending on which translators are available or configured */
static AST_LIST_HEAD_STATIC(g_sangoma_translators, sangoma_translator);

/* \brief list of current sangoma transcoding sessions */
static AST_LIST_HEAD_STATIC(g_sangoma_sessions, sangoma_transcoding_session);

/* \brief Module description */
static char g_sangoma_module_description[] = "Sangoma Codec Translator";

/* \brief cli handler to show sangoma transcoding usage */
#if ASTERISK_VERSION_NUM >= 10600
static char *handle_cli_sangoma_show_translators(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
static char *handle_cli_sangoma_show_transcoding(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
static char *handle_cli_sangoma_flush_transcoding(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
static char *handle_cli_sangoma_show_rtp_statistics(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
static char *handle_cli_sangoma_record_rtp(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
#else
int handle_cli_sangoma_show_translators(int fd, int argc, char *argv[]);
int handle_cli_sangoma_show_transcoding(int fd, int argc, char *argv[]);
int handle_cli_sangoma_flush_transcoding(int fd, int argc, char *argv[]);
int handle_cli_sangoma_show_rtp_statistics(int fd, int argc, char *argv[]);
#endif

static char g_cli_sangoma_show_translators_usage[] = 
			"Usage: sangoma show translators\n"
			"	Display Sangoma registered translators\n";
static char g_cli_sangoma_show_translators_description[] = "Display Sangoma registered translators.";

static char g_cli_sangoma_show_transcoding_usage[] = 
			"Usage: sangoma show transcoding sessions\n"
			"		Display Sangoma current transcoding sessions and statistics\n";
static char g_cli_sangoma_show_transcoding_description[] = "Display Sangoma transcoding utilization.";

static char g_cli_sangoma_flush_transcoding_usage[] = 
			"Usage: sangoma flush transcoding session\n"
			"		Flush all or only the provided sangoma transcoding session RTP buffers\n";
static char g_cli_sangoma_flush_transcoding_description[] = "Flush Sangoma RTP transcoding buffers (be careful).";

static char g_cli_sangoma_show_rtp_usage[] = 
			"Usage: sangoma show rtp statistics\n"
			"		Display Sangoma current RTP statistics per session\n";
static char g_cli_sangoma_show_rtp_description[] = "Display Sangoma RTP utilization.";

#if ASTERISK_VERSION_NUM >= 10600
static char g_cli_sangoma_record_rtp_description[] = "Start/Stop RTP audio recording.";
#endif

/* \brief array of cli entries to be registered with the Asterisk CLI core */
static struct ast_cli_entry g_sangoma_cli[] = {
#if ASTERISK_VERSION_NUM >= 10600
	AST_CLI_DEFINE(handle_cli_sangoma_show_translators, g_cli_sangoma_show_translators_description),
	AST_CLI_DEFINE(handle_cli_sangoma_show_transcoding, g_cli_sangoma_show_transcoding_description),
	AST_CLI_DEFINE(handle_cli_sangoma_flush_transcoding, g_cli_sangoma_flush_transcoding_description),
	AST_CLI_DEFINE(handle_cli_sangoma_show_rtp_statistics, g_cli_sangoma_show_rtp_description),
	AST_CLI_DEFINE(handle_cli_sangoma_record_rtp, g_cli_sangoma_record_rtp_description)
#else
	{
		{ "sangoma", "show", "translators", NULL },
		handle_cli_sangoma_show_translators,
		g_cli_sangoma_show_translators_description,
		g_cli_sangoma_show_translators_usage, NULL, NULL, 
	},
	{
		{ "sangoma", "show", "transcoding", "sessions", NULL },
		handle_cli_sangoma_show_transcoding,
		g_cli_sangoma_show_transcoding_description,
		g_cli_sangoma_show_transcoding_usage, NULL, NULL, 
	},
	{
		{ "sangoma", "flush", "transcoding", NULL },
		handle_cli_sangoma_flush_transcoding,
		g_cli_sangoma_flush_transcoding_description,
		g_cli_sangoma_flush_transcoding_usage, NULL, NULL, 
	},
	{
		{ "sangoma", "show", "rtp", "statistics", NULL },
		handle_cli_sangoma_show_rtp_statistics,
		g_cli_sangoma_show_rtp_description,
		g_cli_sangoma_show_rtp_usage, NULL, NULL, 
	}
	
#endif
};

#if ASTERISK_VERSION_NUM < 10400
AST_MUTEX_DEFINE_STATIC(usemutex);
static int localusecnt = 0;
#define AST_MODULE_LOAD_DECLINE -1
#define AST_MODULE_LOAD_FAILURE -1
#define AST_MODULE_LOAD_SUCCESS 0
#define ast_free free
#define ast_calloc calloc
#define ARRAY_LEN(a) sizeof(a)/sizeof(a[0])
#define ast_vasprintf vasprintf
#endif

#if ASTERISK_VERSION_NUM > 10602
#define sangoma_rtp_new(sess,localip) sess = ast_rtp_instance_new(SANGOMA_DEFAULT_RTP_STACK, g_sched, localip, NULL)
#define sangoma_rtp_write ast_rtp_instance_write
#define sangoma_rtp_read(instance) ast_rtp_instance_read(instance, 0)
#define sangoma_rtp_get_local_address(sess, bindaddr) ast_rtp_instance_get_local_address(sess, bindaddr)
#define sangoma_rtp_destroy ast_rtp_instance_destroy
#define sangoma_rtp_set_remote_address ast_rtp_instance_set_remote_address
#else
#define sangoma_rtp_new(sess,localip) sess = ast_rtp_new_with_bindaddr(g_sched, NULL, 0, 0, localip)
#define sangoma_rtp_write ast_rtp_write
#define sangoma_rtp_read ast_rtp_read
#define sangoma_rtp_get_local_address(sess, bindaddr) ast_rtp_get_us(sess, bindaddr)
#define sangoma_rtp_destroy ast_rtp_destroy
#define sangoma_rtp_set_remote_address ast_rtp_set_peer
#endif

#if ASTERISK_VERSION_NUM < 100000
#define sangoma_src_format_name(trans) ast_getformatname(1 << ((trans)->srcfmt))
#define sangoma_dst_format_name(trans) ast_getformatname(1 << ((trans)->dstfmt))
#define sangoma_src_format_id(trans) (1 << (trans)->srcfmt)
#define sangoma_dst_format_id(trans) (1 << (trans)->dstfmt)
#else
#define sangoma_src_format_name(trans) ast_getformatname(&((trans)->src_format))
#define sangoma_dst_format_name(trans) ast_getformatname(&((trans)->dst_format))
#define sangoma_src_format_id(trans) ((trans)->src_format.id)
#define sangoma_dst_format_id(trans) ((trans)->dst_format.id)
#endif

/* Asterisk 1.6.2 annoying linked list warning,
 * we could check here if ASTERISK_VERSION_NUM 10602,
 * but does not seem to hurt adding this for other versions
 * anyways */
#define QUIET_LINKED_LIST (void)__list_prev

static void sangoma_print_translators(int fd)
{
	int i = 0;
	struct sangoma_translator *cur;
	AST_LIST_LOCK(&g_sangoma_translators);
	AST_LIST_TRAVERSE_SAFE_BEGIN(&g_sangoma_translators, cur, entry) {
		QUIET_LINKED_LIST;
		ast_cli(fd, "%s to %s\n", sangoma_src_format_name(&cur->t), sangoma_dst_format_name(&cur->t));
		i++;
	}
	AST_LIST_TRAVERSE_SAFE_END;
	AST_LIST_UNLOCK(&g_sangoma_translators);
	ast_cli(fd, "Total translators: %d\n", i);
}

static void sangoma_print_transcoding_sessions(int fd)
{
	struct sangoma_transcoding_session *cur;
	AST_LIST_LOCK(&g_sangoma_sessions);
	AST_LIST_TRAVERSE(&g_sangoma_sessions, cur, entry) {
		ast_cli(fd, "[%03llu]%s to %s\tTx Frames: %llu Rx Frames: %llu Rx Nulls %llu Rx Calls %llu AvgRxCallTime %dms AvgRxTime %dms AvgTxTime %dms\n", cur->sessid, 
				sangoma_src_format_name(cur->owner->t), sangoma_dst_format_name(cur->owner->t),
				cur->txcount, cur->rxcount, cur->rxnullcount, cur->rxcallcount, cur->rxcallavgtime, cur->rxavgtime, cur->txavgtime);
	}
	ast_cli(fd, "Total Sessions: %llu\n", g_total_sessions);
	AST_LIST_UNLOCK(&g_sangoma_sessions);
}

#define BREAK_OR_CONTINUE(cursess, sessid) \
	if (sessid >= 0 && (cursess)->sessid == sessid) { \
		break; \
	} else { \
		continue; \
	}
static void sangoma_flush_transcoding_sessions(int fd, int sessid)
{
	struct sangoma_transcoding_session *cur;
	struct ast_frame *f;
	int res;

	if (sessid >= 0 ) {
		ast_cli(fd, "Flushing RTP for session %d only\n", sessid);
	} else {
		ast_log(LOG_WARNING, "Flushing all transcoding session buffers\n");
		ast_cli(fd, "Flushing RTP for all transcoding sessions\n");
	}

	AST_LIST_LOCK(&g_sangoma_sessions);
	AST_LIST_TRAVERSE(&g_sangoma_sessions, cur, entry) {

		if (sessid >= 0 && sessid != cur->sessid) {
			continue;
		}

		res = ast_mutex_lock(&cur->rxrtp->rxlock);
		if (res) {
			ast_log(LOG_ERROR, "Could not acquire rx lock during RTP flush in session %llu: (err %d)\n\n", cur->sessid, res);
			BREAK_OR_CONTINUE(cur, sessid);
		}

		do {
			f = sangoma_rtp_read(cur->rxrtp->ast_rtp);
		} while (f && f->frametype != AST_FRAME_NULL);

		cur->rxcount = 0;
		cur->rxnullcount = 0;
		cur->rxcallavgtime = 0;
		cur->rxavgtime = 0;

		ast_mutex_unlock(&cur->rxrtp->rxlock);

		res = ast_mutex_lock(&cur->txrtp->txlock);
		if (res) {
			ast_log(LOG_ERROR, "Could not acquire tx lock during RTP flush in session %llu: (err %d)\n\n", cur->sessid, res);
			BREAK_OR_CONTINUE(cur, sessid);
		}

		cur->txcount = 0;
		cur->txavgtime = 0;

		ast_mutex_unlock(&cur->txrtp->txlock);

		ast_cli(fd, "[%03llu]%s to %s\tflushed\n", cur->sessid, sangoma_src_format_name(cur->owner->t), sangoma_dst_format_name(cur->owner->t));

		BREAK_OR_CONTINUE(cur, sessid);
	}
	ast_cli(fd, "Total Sessions: %llu\n", g_total_sessions);
	AST_LIST_UNLOCK(&g_sangoma_sessions);
}

#if ASTERISK_VERSION_NUM >= 10600
static void sangoma_record_rtp(int fd, int sessid, int start)
{
	struct sangoma_transcoding_session *cur;
	int res;
	char rtpfilename[255];

	if (start) {
		if (sessid >= 0) {
			ast_cli(fd, "Recording RTP for session %d only\n", sessid);
		} else {
			ast_log(LOG_WARNING, "Recording all transcoding sessions\n");
			ast_cli(fd, "Recording RTP for all transcoding sessions\n");
		}
	}

	AST_LIST_LOCK(&g_sangoma_sessions);
	AST_LIST_TRAVERSE(&g_sangoma_sessions, cur, entry) {

		if (sessid >= 0 && sessid != cur->sessid) {
			continue;
		}

		res = ast_mutex_lock(&cur->rxrtp->rxlock);
		if (res) {
			ast_log(LOG_ERROR, "Could not acquire rx lock during RTP recording in session %llu: (err %d)\n\n", cur->sessid, res);
			BREAK_OR_CONTINUE(cur, sessid);
		}

		if (cur->rxfile) {
			fclose(cur->rxfile);
			cur->rxfile = NULL;
		}

		if (start) {
			snprintf(rtpfilename, sizeof(rtpfilename), "sangoma-rtp-%llu-%sto%s-rx.%s", 
				cur->sessid, sangoma_src_format_name(cur->owner->t), sangoma_dst_format_name(cur->owner->t), sangoma_dst_format_name(cur->owner->t));
			cur->rxfile = fopen(rtpfilename, "w");
			if (!cur->rxfile) {
				ast_mutex_unlock(&cur->rxrtp->rxlock);
				ast_log(LOG_ERROR, "Could not create RTP recording file %s in session %llu: (err %s)\n\n", rtpfilename, cur->sessid, strerror(errno));
				BREAK_OR_CONTINUE(cur, sessid);
			}
			ast_cli(fd, "[%03llu]%s to %s\trx recording started\n", cur->sessid, sangoma_src_format_name(cur->owner->t), sangoma_dst_format_name(cur->owner->t));
		} else {
			ast_cli(fd, "[%03llu]%s to %s\trx recording stopped\n", cur->sessid, sangoma_src_format_name(cur->owner->t), sangoma_dst_format_name(cur->owner->t));
		}

		ast_mutex_unlock(&cur->rxrtp->rxlock);

		if (cur->txfile) {
			fclose(cur->txfile);
			cur->txfile = NULL;
		}

		res = ast_mutex_lock(&cur->txrtp->txlock);
		if (res) {
			ast_log(LOG_ERROR, "Could not acquire tx lock during RTP recording in session %llu: (err %d)\n\n", cur->sessid, res);
			BREAK_OR_CONTINUE(cur, sessid);
		}

		if (start) {
			snprintf(rtpfilename, sizeof(rtpfilename), "sangoma-rtp-%llu-%sto%s-tx.%s", 
				cur->sessid, sangoma_src_format_name(cur->owner->t), sangoma_dst_format_name(cur->owner->t), sangoma_dst_format_name(cur->owner->t));
			cur->txfile = fopen(rtpfilename, "w");
			if (!cur->txfile) {
				ast_mutex_unlock(&cur->txrtp->txlock);
				ast_log(LOG_ERROR, "Could not create RTP recording file %s in session %llu: (err %s)\n\n", rtpfilename, cur->sessid, strerror(errno));
				BREAK_OR_CONTINUE(cur, sessid);
			}
			ast_cli(fd, "[%03llu]%s to %s\ttx recording started\n", cur->sessid, sangoma_src_format_name(cur->owner->t), sangoma_dst_format_name(cur->owner->t));
		} else {
			ast_cli(fd, "[%03llu]%s to %s\ttx recording stopped\n", cur->sessid, sangoma_src_format_name(cur->owner->t), sangoma_dst_format_name(cur->owner->t));
		}

		ast_mutex_unlock(&cur->txrtp->txlock);

		BREAK_OR_CONTINUE(cur, sessid);
	}
	ast_cli(fd, "Total Sessions: %llu\n", g_total_sessions);
	AST_LIST_UNLOCK(&g_sangoma_sessions);
}
#endif

static void sangoma_print_rtp_statistics(int fd)
{
	struct sangoma_transcoding_session *cur;
	AST_LIST_LOCK(&g_sangoma_sessions);
	AST_LIST_TRAVERSE(&g_sangoma_sessions, cur, entry) {
		ast_cli(fd, "[%03llu]%s to %s | MaxUdpQueued: %d | Tx Frame Size: %d | Rx Frame Size: %d | Rx Lost: %li | Rx Ignored: %li | Tx Sock: %d.%d.%d.%d:%d -> %d.%d.%d.%d:%d | Rx Sock: %d.%d.%d.%d:%d <- %d.%d.%d.%d:%d\n",
				cur->sessid, sangoma_src_format_name(cur->owner->t),
				sangoma_dst_format_name(cur->owner->t),
				cur->maxudpqueued, cur->txfrsize, cur->rxfrsize, cur->lostrxpackets, cur->ignoredrxpackets,
				SNGTC_NIPV4(cur->codec_reply.a.host_ip), cur->codec_reply.a.host_udp_port,
				SNGTC_NIPV4(cur->codec_reply.a.codec_ip), cur->codec_reply.a.codec_udp_port,
				SNGTC_NIPV4(cur->codec_reply.b.host_ip), cur->codec_reply.b.host_udp_port,
				SNGTC_NIPV4(cur->codec_reply.b.codec_ip), cur->codec_reply.b.codec_udp_port);
	}
	AST_LIST_UNLOCK(&g_sangoma_sessions);
}

#if ASTERISK_VERSION_NUM >= 10600
static char *handle_cli_sangoma_show_translators(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "sangoma show translators";
		e->usage = g_cli_sangoma_show_translators_usage;
		break;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != 3) {
		return CLI_SHOWUSAGE;
	}
	sangoma_print_translators(a->fd);
	return CLI_SUCCESS;
}

static char *handle_cli_sangoma_show_transcoding(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "sangoma show transcoding sessions";
		e->usage = g_cli_sangoma_show_transcoding_usage;
		break;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != 4) {
		return CLI_SHOWUSAGE;
	}
	sangoma_print_transcoding_sessions(a->fd);
	return CLI_SUCCESS;
}

static char *handle_cli_sangoma_flush_transcoding(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int sessid = -1;
	switch (cmd) {
	case CLI_INIT:
		e->command = "sangoma flush transcoding";
		e->usage = g_cli_sangoma_flush_transcoding_usage;
		break;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc < 3) {
		return CLI_SHOWUSAGE;
	}
	if (a->argc == 4) {
		sessid = atoi(a->argv[3]);
	}
	sangoma_flush_transcoding_sessions(a->fd, sessid);
	return CLI_SUCCESS;
}

static char *handle_cli_sangoma_show_rtp_statistics(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "sangoma show rtp statistics";
		e->usage = g_cli_sangoma_show_rtp_usage;
		break;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != 4) {
		return CLI_SHOWUSAGE;
	}
	sangoma_print_rtp_statistics(a->fd);
	return CLI_SUCCESS;
}

static char *handle_cli_sangoma_record_rtp(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int sessid = -1;
	int start = 0;
	switch (cmd) {
	case CLI_INIT:
		e->command = "sangoma record rtp [yes|no]";
		e->usage = g_cli_sangoma_show_rtp_usage;
		break;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc < 4) {
		return CLI_SHOWUSAGE;
	}

	if (ast_true(a->argv[3])) {
		start = 1;
	}

	if (a->argc == 5) {
		sessid = atoi(a->argv[4]);
	}
	sangoma_record_rtp(a->fd, sessid, start);
	return CLI_SUCCESS;
}
#else
int handle_cli_sangoma_show_translators(int fd, int argc, char *argv[])
{
	sangoma_print_translators(fd);
	return 0;
}

int handle_cli_sangoma_show_transcoding(int fd, int argc, char *argv[])
{
	sangoma_print_transcoding_sessions(fd);
	return 0;
}

int handle_cli_sangoma_flush_transcoding(int fd, int argc, char *argv[])
{
	int sessid = -1;
	if (argc == 4) {
		sessid = atoi(argv[3]);
	}
	sangoma_flush_transcoding_sessions(fd, sessid);
	return 0;
}

int handle_cli_sangoma_show_rtp_statistics(int fd, int argc, char *argv[])
{
	sangoma_print_rtp_statistics(fd);
	return 0;
}
#endif

static struct ast_frame *sangoma_fakesrc_sample(void)
{
	static struct ast_frame f = {
		.frametype = AST_FRAME_VOICE,
		.samples = SANGOMA_DEFAULT_SAMPLES,
		.src = __PRETTY_FUNCTION__
	};
	return &f;
}

#if ASTERISK_VERSION_NUM < 100000
#define sangoma_cmp_src_format(frame, trans) ((frame)->subclass_integer != (1 << ((trans)->srcfmt)))
#define sangoma_cmp_dst_format(frame, trans) ((frame)->subclass_integer != (1 << ((trans)->dstfmt)))
#define sangoma_frame_format_name(frame) ast_getformatname((frame)->subclass_integer)
#else
#define sangoma_cmp_src_format(frame, trans) ast_format_cmp(&(frame)->subclass.format, &(trans)->src_format)
#define sangoma_cmp_dst_format(frame, trans) ast_format_cmp(&(frame)->subclass.format, &(trans)->dst_format)
#define sangoma_frame_format_name(frame) ast_getformatname(&(frame)->subclass.format)
#endif

/*! \brief enqueue input frame to sangoma session */
static int sangoma_framein(struct asterisk_transcoding_pvt *pvt, struct ast_frame *f)
{
	int difftime = 0;
	int res = 0;
	struct timeval now = {0, 0};
	struct sangoma_transcoding_session *sess = sangoma_get_pvt(pvt);

	if (!f->subclass_integer) {
		/* we must return a fake frame for calculation purposes, 
		 * may be we should indeed send something to sangoma board? */
		sess->fakestate = FAKE_FRAME_REQUESTED;
		pvt->samples = f->samples;
		return 0;
	}

	if (f->frametype != AST_FRAME_VOICE) {
		SANG_SESS_WARN_TX("refusing to write frame of type %d to Sangoma transcoder %llu, we only accept voice in format %s\n", 
				f->frametype, sess->sessid, sangoma_src_format_name(pvt->t));
		return -1;
	}

	if (f->frametype != AST_FRAME_VOICE || sangoma_cmp_src_format(f, pvt->t)) {
		SANG_SESS_WARN_TX("refusing to write voice frame in format %s to Sangoma transcoder %llu, we only accept voice in format %s\n", 
				sangoma_frame_format_name(f), sess->sessid, sangoma_src_format_name(pvt->t));
		return -1;
	}

	res = ast_mutex_trylock(&sess->txrtp->txlock);
	if (res) {
		ast_log(LOG_ERROR, "Could not acquire tx lock for RTP tx in session %llu: (err %d)\n", sess->sessid, res);
		return -1;
	}

	if (!sess->txfrsize) {
		sess->txfrsize = f->datalen;
	} else if (sess->txfrsize != f->datalen) {
		/*ast_log(LOG_DEBUG, "Changing tx frame size from %d to %d in session %llu\n", sess->txfrsize, f->datalen, sess->sessid);*/
		sess->txfrsize = f->datalen;
	}

	if (sangoma_rtp_write(sess->txrtp->ast_rtp, f)) {
		SANG_SESS_WARN_TX("Failed to write voice frame in format %s of length %d to Sangoma transcoder %llu\n", 
		sangoma_frame_format_name(f), f->datalen, sess->sessid);
		ast_mutex_unlock(&sess->txrtp->txlock);
		return -1;
	}

	if (sess->txfile) {
#if ASTERISK_VERSION_NUM >= 10601
		fwrite(f->data.ptr, 1, f->datalen, sess->txfile);
#else
		fwrite(f->data, 1, f->datalen, sess->txfile);
#endif
	}

	ast_mutex_unlock(&sess->txrtp->txlock);
	
	now = ast_tvnow();
	if (!sess->lasttxtime.tv_sec) {
		sess->lasttxtime = now;
	} else {
		difftime = ast_tvdiff_ms(now, sess->lasttxtime);
		sess->txavgtime = sess->txavgtime ? ((sess->txavgtime + difftime) / 2) : difftime;
		sess->lasttxtime = now;
	}
	sess->txcount++;

	pvt->samples += f->samples;

	return 0;
}

/*! \brief output frames from the sangoma session */
static struct ast_frame *sangoma_frameout(struct asterisk_transcoding_pvt *pvt)
{
	/* read from the rtp session and put the result in pvt->f */
	int difftime = 0;
	int res = 0;
	int windex = 0;
	int wrapped = 0;
	int i = 0;
	int udpqueued_bytes = 0;
	int udpqueued = 0;
	struct frame_list rtp_frames;
	struct ast_frame *rtp_f = NULL;
	struct ast_frame *f = NULL;
	struct timeval now = {0, 0};
	struct ast_frame *rtp_queue[4];
	struct sangoma_transcoding_session *sess = sangoma_get_pvt(pvt);

	memset(rtp_queue, 0, sizeof(rtp_queue));

	if (FAKE_FRAME_REQUESTED == sess->fakestate) {
		sess->fakestate = FAKE_FRAME_PROVIDED;
		pvt->f.frametype = AST_FRAME_VOICE;
		pvt->f.subclass_integer = 0;
		pvt->f.samples = SANGOMA_DEFAULT_SAMPLES;
/* if any version previous to 1.6.1 defines data.ptr, then decrease this version num */
#if ASTERISK_VERSION_NUM >= 10800
		pvt->f.data.ptr = NULL;
#elif ASTERISK_VERSION_NUM >= 10601
		pvt->f.data.ptr = NULL;
#if ASTERISK_VERSION_NUM >= 10602
		ast_set_flag(&pvt->f, AST_FRFLAG_FROM_TRANSLATOR);
#endif
#else
		pvt->f.data = NULL;
#endif
		pvt->f.offset = 0;
		pvt->f.datalen = 0;
		pvt->f.mallocd = 0;
		pvt->samples = 0;
		/* we fake some processing to not get an awesome rate, we only want straight transcoding requests */
		for (res = 0; res < INT_MAX/1000; res++) {
			difftime++;
		}
#if ASTERISK_VERSION_NUM == 10601 || ASTERISK_VERSION_NUM >= 10800
		return ast_frisolate(&pvt->f);
#else
		return &pvt->f;
#endif
	} else if (FAKE_FRAME_PROVIDED == sess->fakestate) {
		/*ast_log(LOG_DEBUG, "Terminated fake frame request to Sangoma transcoder\n");*/
		sess->fakestate = FAKE_FRAME_TERMINATED;
		return NULL;
	}

	res = ast_mutex_trylock(&sess->rxrtp->rxlock);
	if (res) {
		ast_log(LOG_ERROR, "Could not acquire rx lock for RTP rx in session %llu: (err %d)\n\n", sess->sessid, res);
		return NULL;
	}

	AST_LIST_HEAD_INIT_NOLOCK(&rtp_frames);

	if (!sess->rxcount) {
		/* if this is the first time we read we must flush it */
		for ( ; ; ) {
			f = sangoma_rtp_read(sess->rxrtp->ast_rtp);
			if (f && f->frametype != AST_FRAME_NULL) {
				if (rtp_f) {
					ast_frfree(rtp_f);
				}
				rtp_f = ast_frdup(f);
				continue;
			}
			break;
		}
		AST_LIST_INSERT_TAIL(&rtp_frames, rtp_f, frame_list);
		sess->rxcount++;
	} else {
		for ( ; ; ) {
			f = sangoma_rtp_read(sess->rxrtp->ast_rtp);
			if (f && f->frametype == AST_FRAME_VOICE) {

				if (sangoma_cmp_dst_format(f, pvt->t)) {
					SANG_SESS_WARN_RX("ignoring read voice frame in format %s from Sangoma transcoder %llu, we only expect voice in format %s rxrtp=%p\n",
							sangoma_frame_format_name(f), sess->sessid, sangoma_dst_format_name(pvt->t), sess->rxrtp);
					break;
				}
				udpqueued++;
				rtp_f = ast_frdup(f);

				if (windex == ARRAY_LEN(rtp_queue)) {
					windex = 0;
					wrapped = 1;
				}

				if (rtp_queue[windex]) {
					sess->ignoredrxpackets++;
					ast_frfree(rtp_queue[windex]);
				}
				rtp_queue[windex] = rtp_f;
				windex++;
		
				sess->rxcount++;
				if (!sess->rxfrsize) {
					sess->rxfrsize = rtp_f->datalen;
				} else if (sess->rxfrsize != rtp_f->datalen) {
					ast_log(LOG_WARNING, "Changing rx frame size from %d to %d in session %llu\n", sess->rxfrsize, f->datalen, sess->sessid);
					sess->rxfrsize = rtp_f->datalen;
				}

				/* check sequence */
				if (sess->lastrxseqno >= 0) {
					int expectedseq = sess->lastrxseqno + 1;
					if (rtp_f->seqno > expectedseq) {
						int dropped = (rtp_f->seqno - expectedseq);
						SANG_SESS_WARN_RX("[%03llu][%sto%s] Got Seq %d but expecting %d (time since last read = %dms), dropped %d packets\n", sess->sessid, 
								sangoma_src_format_name(sess->owner->t), sangoma_dst_format_name(sess->owner->t), 
								rtp_f->seqno, expectedseq, difftime, dropped);
						sess->lostrxpackets += dropped;
					} else if (rtp_f->seqno < expectedseq && rtp_f->seqno) {
						ast_log(LOG_WARNING, "[%03llu][%sto%s] Got backward Seq %d, expecting %d (time since last read = %dms)\n", sess->sessid, 
								sangoma_src_format_name(sess->owner->t), sangoma_dst_format_name(sess->owner->t), 
								rtp_f->seqno, expectedseq, difftime);
					}
				}

				if (sess->lostrxpackets == -1) {
					sess->lostrxpackets = 0;
				}

				sess->lastrxseqno = rtp_f->seqno; 
				continue;
			}
			break;
		}
		if (wrapped) {
			for (i = windex; i < ARRAY_LEN(rtp_queue); i++) {
				udpqueued_bytes += rtp_queue[i]->datalen;
				AST_LIST_INSERT_TAIL(&rtp_frames, rtp_queue[i], frame_list);
				if (sess->rxfile) {
#if ASTERISK_VERSION_NUM >= 10601
					fwrite(rtp_queue[i]->data.ptr, 1, rtp_queue[i]->datalen, sess->rxfile);
#else
					fwrite(rtp_queue[i]->data, 1, rtp_queue[i]->datalen, sess->rxfile);
#endif
				}
			}
		}
		for (i = 0; i < windex; i++) {
			udpqueued_bytes += rtp_queue[i]->datalen;
			AST_LIST_INSERT_TAIL(&rtp_frames, rtp_queue[i], frame_list);
			if (sess->rxfile) {
#if ASTERISK_VERSION_NUM >= 10601
				fwrite(rtp_queue[i]->data.ptr, 1, rtp_queue[i]->datalen, sess->rxfile);
#else
				fwrite(rtp_queue[i]->data, 1, rtp_queue[i]->datalen, sess->rxfile);
#endif
			}
		}
	}

	/* use the head of the rtp frame list if any */
	if (!AST_LIST_EMPTY(&rtp_frames)) {
		if (sess->maxudpqueued < udpqueued) {
			sess->maxudpqueued = udpqueued;
		}
		f = AST_LIST_FIRST(&rtp_frames);
		if (udpqueued > 1) {
			int pn = 0;
			struct ast_frame *curf;
			//ast_log(LOG_WARNING, "smoothing %d frames into one of %d bytes\n", udpqueued, udpqueued_bytes);
			if (!sess->rxsmoother) {
				sess->rxsmoother = ast_smoother_new(udpqueued_bytes);
				if (!sess->rxsmoother) {
					ast_log(LOG_ERROR, "Failed to create smoother of %d bytes\n", udpqueued_bytes);
					goto framedone;
				}
			} else {
#if ASTERISK_VERSION_NUM > 10423
				ast_smoother_reconfigure(sess->rxsmoother, udpqueued_bytes);
#else
				ast_smoother_reset(sess->rxsmoother, udpqueued_bytes);
#endif
			}

			for (curf = f; curf; curf = AST_LIST_NEXT(curf, frame_list)) {
				if (ast_smoother_feed(sess->rxsmoother, curf)) {
					ast_log(LOG_ERROR, "Failed to feed smoother with frame no %d of %d bytes of a total of %d in session %lld\n", pn+1, curf->datalen, udpqueued_bytes, sess->sessid);
					break;
				}
				pn++;
			}

			/* free the list of frames */
			ast_frfree(f);

			f = ast_smoother_read(sess->rxsmoother);
			if (f) {
				//ast_log(LOG_WARNING, "done with smoothed frame of %d bytes\n", f->datalen);
				f = ast_frdup(f);
			} else {
				ast_log(LOG_ERROR, "fed %d bytes to smoother and did not get a frame back!\n", udpqueued_bytes);
			}
		} else {
			/* twiddle ... nothing to smooth, just use the same frame returned from the RTP stack */
		}
	}

framedone:

	/* calculate average function call time */
	now = ast_tvnow();
	if (!sess->lastrxcalltime.tv_sec) {
		sess->lastrxcalltime = now;
	} else {
		difftime = ast_tvdiff_ms(now, sess->lastrxcalltime);
		sess->rxcallavgtime = sess->rxcallavgtime ? ((sess->rxcallavgtime + difftime) / 2) : difftime;
		sess->lastrxcalltime = now;
	}
	sess->rxcallcount++;

	ast_mutex_unlock(&sess->rxrtp->rxlock);

	if (!f) {
		SANG_SESS_WARN_RX("ignoring read NULL frame from Sangoma transcoder %llu, we only expect voice in format %s\n", 
				sess->sessid, sangoma_dst_format_name(pvt->t));
		return NULL;
	}

	if (f->frametype == AST_FRAME_NULL) {
		sess->rxnullcount++;
		ast_frfree(f);
		return NULL;
	}
	if (f->frametype != AST_FRAME_VOICE) {
		SANG_SESS_WARN_RX("ignoring read frame of type %d from Sangoma transcoder %llu, we only expect voice in format %s\n", 
				f->frametype, sess->sessid, sangoma_dst_format_name(pvt->t));
		ast_frfree(f);
		return NULL;
	}

	/* calculate average read time */
	now = ast_tvnow();
	if (!sess->lastrxtime.tv_sec) {
		sess->lastrxtime = now;
	} else {
		difftime = ast_tvdiff_ms(now, sess->lastrxtime);
		sess->rxavgtime = sess->rxavgtime ? ((sess->rxavgtime + difftime) / 2) : difftime;
		sess->lastrxtime = now;
	}


	/* update samples */
	pvt->samples = 0;
	return f;
}

static int sangoma_create_rtp_port(void *usr_priv, uint32_t host_ip, uint32_t *p_rtp_port, void **rtp_fd)
{
	struct sangoma_rtp_handle *sng_rtp;
	struct sockaddr_in localaddr;
#if ASTERISK_VERSION_NUM >= 10800
	struct ast_sockaddr *local_ip;
	struct ast_sockaddr local_ip_s;
#else
	struct in_addr local_ip = { 0 };
#endif
	void *fd;

#if ASTERISK_VERSION_NUM >= 10800
	localaddr.sin_family = AF_INET;
	localaddr.sin_addr.s_addr = htonl(host_ip);
	localaddr.sin_port = 0;
	ast_sockaddr_from_sin(&local_ip_s, &localaddr);
	local_ip = &local_ip_s;
#else
	local_ip.s_addr = htonl(host_ip);
#endif

	/* ask to create the session with the local IP provided */
	ast_log(LOG_DEBUG, "Creating Asterisk RTP with IP %d.%d.%d.%d\n", SNGTC_NIPV4(host_ip));
	sangoma_rtp_new(fd, local_ip);
	if (!fd) {
		ast_log(LOG_ERROR, "Failed to create Sangoma transcoding RTP stream\n");
		return -1;
	}

#if ASTERISK_VERSION_NUM >= 10800
	sangoma_rtp_get_local_address(fd, &local_ip_s);
	ast_sockaddr_to_sin(&local_ip_s, &localaddr);
#else
	sangoma_rtp_get_local_address(fd, &localaddr);
#endif

	sng_rtp = ast_calloc(1, sizeof(*sng_rtp));
	if (!sng_rtp) {
		ast_log(LOG_ERROR, "Failed to create Sangoma transcoding RTP stream handle\n");
		sangoma_rtp_destroy(fd);
		return -1;
	}
	sng_rtp->ast_rtp = fd;
	
	ast_mutex_init(&sng_rtp->rxlock);
	ast_mutex_init(&sng_rtp->txlock);

	*p_rtp_port = ntohs(localaddr.sin_port);
	*rtp_fd = sng_rtp;
	ast_log(LOG_DEBUG, "Created Asterisk RTP %p\n", fd);
	return 0;
	
}

static int sangoma_create_rtp(void *usr_priv, sngtc_codec_request_leg_t *codec_reg_leg, sngtc_codec_reply_leg_t* codec_reply_leg, void **rtp_fd)
{
	struct sangoma_rtp_handle *sng_rtp;
	struct sockaddr_in remoteaddr;
#if ASTERISK_VERSION_NUM >= 10800
	struct ast_sockaddr remoteaddr_s;
#endif

	sng_rtp = *rtp_fd;

	/* set the socket where we will send the data to and read from */
	remoteaddr.sin_family = AF_INET;
	remoteaddr.sin_addr.s_addr = htonl(codec_reply_leg->codec_ip);
	remoteaddr.sin_port = htons(codec_reply_leg->codec_udp_port);

#if ASTERISK_VERSION_NUM >= 10800
	ast_sockaddr_from_sin(&remoteaddr_s, &remoteaddr);
	sangoma_rtp_set_remote_address(sng_rtp->ast_rtp, &remoteaddr_s);
#else
	sangoma_rtp_set_remote_address(sng_rtp->ast_rtp, &remoteaddr);
#endif
 
	ast_log(LOG_DEBUG, "Creating Codec on host (0x%08X/%d)  codec(0x%08X/%d) CODEC ID=%i ms=%i FD=%p \n",
				codec_reg_leg->host_ip, codec_reg_leg->host_udp_port,
				codec_reply_leg->codec_ip, codec_reply_leg->codec_udp_port, 
				codec_reg_leg->codec_id,  codec_reg_leg->ms, sng_rtp->ast_rtp);

	return 0;
}

static int sangoma_destroy_rtp(void *usr_priv, void *fd)
{
	struct sangoma_rtp_handle *rtp = fd;
	if (!fd) {
		return 0;
	}
	ast_log(LOG_DEBUG, "Destroying Asterisk RTP %p\n", rtp->ast_rtp);
	sangoma_rtp_destroy(rtp->ast_rtp);
	rtp->ast_rtp = NULL;
	ast_mutex_destroy(&rtp->rxlock);
	ast_mutex_destroy(&rtp->txlock);
	rtp->flags = 0;
	ast_free(rtp);
	return 0;
}

static int sangoma_release_rtp_port(void *usr_priv, uint32_t host_ip, uint32_t p_rtp_port, void *rtp_fd)
{
	if (rtp_fd) {
		sangoma_destroy_rtp(usr_priv, rtp_fd);
	}
	return 0;
}

static int ast_format_to_codec_id(sangoma_format_id astformat, int *ptime)
{
	int i;
	for (i = 0; i < sng_array_len(g_codec_map); i++) {
		if (g_codec_map[i].compat_format == astformat) {
			*ptime = g_codec_map[i].maxms;
			return g_codec_map[i].codec_id;
		}
	}
	return -1;
}

#if 0
static void print_stack()
{
	void *trace[50];
	char **symbols;
	int rc, i;

	rc = backtrace(trace, sizeof(trace)/sizeof(trace[0]));
	symbols = backtrace_symbols(trace, rc);
	for (i = 0; i < rc; i++) {
		ast_log(LOG_DEBUG, "%s\n", symbols[i]);
	}
	free(symbols);
}
#endif

/* \brief initialize the new sangoma sangoma transcoding session */
static int sangoma_new(struct asterisk_transcoding_pvt *pvt)
{
	int err = 0;
	int codec_id = -1;
	int ptime = 0;
	struct sangoma_transcoding_session *sess = sangoma_get_pvt(pvt);

	/* set a reference to our owner */
	sess->owner = pvt;
	sess->lostrxpackets = -1;
	sess->lastrxseqno = -1;

	sess->codec_request.usr_priv = sess;

	ast_log(LOG_DEBUG, "Got codec request from Asterisk core from %s to %s\n", sangoma_src_format_name(pvt->t), sangoma_dst_format_name(pvt->t));

	/*print_stack();*/

	codec_id = ast_format_to_codec_id(sangoma_src_format_id(pvt->t), &ptime);
	if (codec_id == -1) {
		ast_log(LOG_ERROR, "Failed to map source ast format %d to codec id\n", sangoma_src_format_id(pvt->t));
		return -1;
	}
	sess->codec_request.a.host_ip = g_rtpip;
	sess->codec_request.a.codec_id = codec_id;
	sess->codec_request.a.ms = ptime;

	codec_id = ast_format_to_codec_id(sangoma_dst_format_id(pvt->t), &ptime);
	if (codec_id == -1) {
		ast_log(LOG_ERROR, "Failed to map source ast format %d to codec id\n", sangoma_dst_format_id(pvt->t));
		return -1;
	}
	sess->codec_request.b.host_ip = g_rtpip;
	sess->codec_request.b.codec_id = codec_id;
	sess->codec_request.b.ms = ptime;


	ast_log(LOG_DEBUG, "Creating Codec A-CODEC_ID=%i  Codec B-CODEC_ID=%i\n",
				sess->codec_request.a.codec_id, sess->codec_request.b.codec_id);

	/* we re-use the global sessions lock to lock the session request in the library */
	AST_LIST_LOCK(&g_sangoma_sessions);

	/* create the actual new session */
	err = sngtc_create_transcoding_session(&sess->codec_request, &sess->codec_reply, 0);
	if (err) {
		AST_LIST_UNLOCK(&g_sangoma_sessions);
		ast_log(LOG_ERROR, "Failed to create Sangoma transcoding session\n");
		sess->rxrtp = NULL;
		sess->txrtp = NULL;
		return -1;
	}

	/* add the new session to the global list for usage statistics */
	sess->txrtp = sess->codec_reply.tx_fd;
	sess->rxrtp = sess->codec_reply.rx_fd;

	/* sanity check flags, we should not get RTP already used, if this fail is a bug in the library */
	if (ast_test_flag(sess->txrtp, SNG_RTP_TX)) {
		ast_log(LOG_ERROR, "Invalid transcoding session created! tx rtp %p it's already owned for transmission by session %llu!\n", sess->txrtp, sess->txrtp->txsess->sessid);
		sess->rxrtp = NULL;
		sess->txrtp = NULL;
		if (sngtc_free_transcoding_session(&sess->codec_reply)) {
			ast_log(LOG_ERROR, "Failed to destroy Sangoma transcoding session properly (after discovering TX is already in use)\n");
		}
		AST_LIST_UNLOCK(&g_sangoma_sessions);
		return -1;
	}

	if (ast_test_flag(sess->rxrtp, SNG_RTP_RX)) {
		ast_log(LOG_ERROR, "Invalid transcoding session created! rx rtp %p it's already owned for reception by session %llu!\n", sess->rxrtp, sess->rxrtp->rxsess->sessid);
		sess->rxrtp = NULL;
		sess->txrtp = NULL;
		if (sngtc_free_transcoding_session(&sess->codec_reply)) {
			ast_log(LOG_ERROR, "Failed to destroy Sangoma transcoding session properly (after discovering RX is already in use)\n");
		}
		AST_LIST_UNLOCK(&g_sangoma_sessions);
		return -1;
	}

	ast_set_flag(sess->txrtp, SNG_RTP_TX);
	ast_set_flag(sess->rxrtp, SNG_RTP_RX);
	sess->txrtp->txsess = sess;
	sess->rxrtp->rxsess = sess;

	sess->sessid = g_next_session_id++;
	g_total_sessions++;
	
	ast_log(LOG_DEBUG, "Created Codec ID=%llu A-CODEC_ID/IANA=%i/%i  Codec B-CODEC_ID/IANA=%i/%i  Tx=%p  Rx=%p\n",
				sess->sessid,
				sess->codec_request.a.codec_id, 
				sess->codec_reply.a.iana_code,
				sess->codec_request.b.codec_id,
				sess->codec_reply.b.iana_code,
				sess->txrtp,
				sess->rxrtp);

	AST_LIST_INSERT_HEAD(&g_sangoma_sessions, sess, entry);

	AST_LIST_UNLOCK(&g_sangoma_sessions);

	ast_log(LOG_DEBUG, "Sangoma Transcoding Session %llu Created\n", sess->sessid);

	ast_log(LOG_DEBUG, "Sangoma Transcoding Session %llu Rx %s in %d.%d.%d.%d:%d from board at %d.%d.%d.%d:%d fd=%p\n",
			sess->sessid, sangoma_dst_format_name(sess->owner->t),
			SNGTC_NIPV4(sess->codec_reply.b.host_ip), sess->codec_reply.b.host_udp_port,
			SNGTC_NIPV4(sess->codec_reply.b.codec_ip), sess->codec_reply.b.codec_udp_port, sess->rxrtp);

	ast_log(LOG_DEBUG, "Sangoma Transcoding Session %llu Tx %s from %d.%d.%d.%d:%d to board at %d.%d.%d.%d:%d fd=%p\n",
			sess->sessid, sangoma_src_format_name(sess->owner->t),
			SNGTC_NIPV4(sess->codec_reply.a.host_ip), sess->codec_reply.a.host_udp_port,
			SNGTC_NIPV4(sess->codec_reply.a.codec_ip), sess->codec_reply.a.codec_udp_port, sess->txrtp);
	
	return 0;
}

static void sangoma_destroy(struct asterisk_transcoding_pvt *pvt)
{
	struct sangoma_transcoding_session *sess = sangoma_get_pvt(pvt);

	ast_log(LOG_DEBUG, "Destroying Sangoma transcoding session %llu\n", sess->sessid);

	AST_LIST_LOCK(&g_sangoma_sessions);
	
	/* clear sanity flags, we're no longer using this RTP session for tx/rx */
	ast_clear_flag(sess->txrtp, SNG_RTP_TX);
	ast_clear_flag(sess->rxrtp, SNG_RTP_RX);
	sess->txrtp->txsess = NULL;
	sess->rxrtp->rxsess = NULL;

	if (sngtc_free_transcoding_session(&sess->codec_reply)) {
		ast_log(LOG_ERROR, "Failed to destroy Sangoma transcoding session %llu properly\n", sess->sessid);
	}

	if (sess->rxfile) {
		fclose(sess->rxfile);
		sess->rxfile = NULL;
	}

	if (sess->txfile) {
		fclose(sess->txfile);
		sess->txfile = NULL;
	}

	if (sess->rxsmoother) {
		ast_smoother_free(sess->rxsmoother);
	}

	AST_LIST_REMOVE(&g_sangoma_sessions, sess, entry);
	g_total_sessions--;
	AST_LIST_UNLOCK(&g_sangoma_sessions);
}

#if ASTERISK_VERSION_NUM < 10400
#define LOCAL_COUNT_INC  \
	ast_mutex_lock(&usemutex); \
	localusecnt++; \
	ast_mutex_unlock(&usemutex); 
#define LOCAL_COUNT_DEC  \
	ast_mutex_lock(&usemutex); \
	localusecnt--; \
	ast_mutex_unlock(&usemutex); 
static struct asterisk_transcoding_pvt *sangoma_new_g729toulaw()
{
	struct asterisk_transcoding_pvt *pvt = NULL;
	LOCAL_COUNT_INC;
	pvt = calloc(1, sizeof(struct asterisk_transcoding_pvt));
	if (!pvt) {
		ast_log(LOG_ERROR, "Could not allocate pvt structure\n");
		LOCAL_COUNT_DEC;
		return NULL;
	}
	pvt->dstfmt = 2;
	pvt->srcfmt = 8;
	pvt->t = pvt;
	if (sangoma_new(pvt)) {
		free(pvt);
		LOCAL_COUNT_DEC;
		return NULL;
	}
	return pvt;
}

static struct asterisk_transcoding_pvt *sangoma_new_ulawtog729()
{
	struct asterisk_transcoding_pvt *pvt = NULL;
	LOCAL_COUNT_INC;
	pvt = calloc(1, sizeof(struct asterisk_transcoding_pvt));
	if (!pvt) {
		LOCAL_COUNT_DEC;
		ast_log(LOG_ERROR, "Could not allocate pvt structure\n");
		return NULL;
	}
	pvt->dstfmt = 8;
	pvt->srcfmt = 2;
	pvt->t = pvt;
	if (sangoma_new(pvt)) {
		free(pvt);
		LOCAL_COUNT_DEC;
		return NULL;
	}
	return pvt;
}

static void sangoma_destroy_pvt_old(struct asterisk_transcoding_pvt *pvt)
{
	sangoma_destroy(pvt);
	free(pvt);
	LOCAL_COUNT_DEC;
}
#endif /* endif ASTERISK 1.2 */

/* updates the list of codecs and return the number of codecs/translators added */
static unsigned int update_codec_list(char codec_list[MAX_CODECS_LIST][MAX_CODEC_STR], size_t size, int index, const char *value)
{
	unsigned int i = index;
	char line[1024];
	char *c = NULL;
	char *tok = NULL;

	ast_log(LOG_DEBUG, "Parsing codec string %s\n", value);
	ast_copy_string(line, value, sizeof(line));

	/* the value string is in format codec,codec,codec where codec can either be a codec name 
	 * or a translator name (translator names are codec1tocodec2 names ) */
	tok = strtok_r(line, ",", &c);
	if (tok == NULL) {
		ast_log(LOG_WARNING, "not found any codec specification!\n");
		return 0;
	}

	ast_copy_string(codec_list[i], tok, sizeof(codec_list[0]));
	ast_log(LOG_NOTICE, "found translator %s\n", codec_list[i]);
	i++;

	while ((tok = strtok_r(NULL, ",", &c))) {
		ast_copy_string(codec_list[i], tok, sizeof(codec_list[0]));
		ast_log(LOG_NOTICE, "found translator %s\n", codec_list[i]);
		i++;
	}

	return (i - index);
}

static int sangoma_parse_config(int reload)
{
	struct ast_variable *v = NULL;
	char *category = NULL;
	struct sockaddr_in rtpaddr;
	struct sockaddr_in nulladdr;
#if ASTERISK_VERSION_NUM >= 10800
	struct ast_sockaddr rtpaddr_s;
	struct ast_sockaddr nulladdr_s;
#endif
	int ret = 0;
#if ASTERISK_VERSION_NUM >= 10600
	struct ast_flags config_flags = { reload ? CONFIG_FLAG_FILEUNCHANGED : 0 };
	struct ast_config *cfg = ast_config_load(SANGOMA_TRANSCODE_CONFIG, config_flags);
	if (cfg == CONFIG_STATUS_FILEUNCHANGED) {
		ast_verbose("File " SANGOMA_TRANSCODE_CONFIG " was not changed, ignoring reload\n");
		return 0;
	}
#else
	struct ast_config *cfg = ast_config_load(SANGOMA_TRANSCODE_CONFIG);
#endif

#if ASTERISK_VERSION_NUM > 10602
	if (cfg == CONFIG_STATUS_FILEMISSING || cfg == CONFIG_STATUS_FILEINVALID) {
		ast_log(LOG_WARNING, "Sangoma transcoding configuration file " SANGOMA_TRANSCODE_CONFIG " is missing or invalid\n");
		return -1;
	}
#endif
	if (!cfg) {
		ast_log(LOG_WARNING, "Failed to Sangoma transcoding configuration file " SANGOMA_TRANSCODE_CONFIG ", make sure the file exists and is readable\n");
		return -1;
	}

	memset(&g_init_cfg, 0, sizeof(g_init_cfg));
	memset(&rtpaddr, 0, sizeof(rtpaddr));
	memset(&nulladdr, 0, sizeof(nulladdr));
#if ASTERISK_VERSION_NUM >= 10800
	memset(&rtpaddr_s, 0, sizeof(rtpaddr_s));
	memset(&nulladdr_s, 0, sizeof(nulladdr_s));
#endif
	memset(g_codec_register_list, 0, sizeof(g_codec_register_list));
	memset(g_codec_noregister_list, 0, sizeof(g_codec_noregister_list));
	while ((category = ast_category_browse(cfg, category))) {

		if (!strcasecmp("general", category)) {
			for (v = ast_variable_browse(cfg, category); v; v = v->next) {
				if (!strcasecmp(v->name, "unregisterconflictingcodecs")) {
					g_attempt_unload_conflicting_translators = ast_true(v->value);
					if (g_attempt_unload_conflicting_translators) {
#if ASTERISK_VERSION_NUM < 10602
						ast_log(LOG_NOTICE, "unregisterconflictingcodecs is now enabled, this may cause warnings when unloading other codec modules (see https://issues.asterisk.org/view.php?id=17092)\n");
#else
						ast_log(LOG_WARNING, "unregisterconflictingcodecs is now enabled, this may cause warnings when unloading other codec modules. This option should not be needed for Asterisk >= 1.6.2\n");
#endif
					}
				} else if (!strcasecmp(v->name, "allownondahditiming")) {
					g_allow_nondahdi_timing = ast_true(v->value);
					if (g_allow_nondahdi_timing) {
						ast_log(LOG_WARNING, "BE AWARE THAT NON-DAHDI TIMING HAS BEEN KNOWN TO CAUSE AUDIO LOSS AND/OR CRASHES\n");
						ast_log(LOG_WARNING, "BE AWARE THAT NON-DAHDI TIMING HAS BEEN KNOWN TO CAUSE AUDIO LOSS AND/OR CRASHES\n");
						ast_log(LOG_WARNING, "BE AWARE THAT NON-DAHDI TIMING HAS BEEN KNOWN TO CAUSE AUDIO LOSS AND/OR CRASHES\n");
					}
				} else if (!strcasecmp(v->name, "register")) {
					g_codec_register_list_count += update_codec_list(g_codec_register_list, ARRAY_LEN(g_codec_register_list), g_codec_register_list_count, v->value);
				} else if (!strcasecmp(v->name, "noregister")) {
					g_codec_noregister_list_count += update_codec_list(g_codec_noregister_list, ARRAY_LEN(g_codec_noregister_list), g_codec_noregister_list_count, v->value);
				} else if (!strcasecmp(v->name, "soapserver")) {
					snprintf(g_str_soap_addr, sizeof(g_str_soap_addr), "%s", v->value);
					ast_log(LOG_NOTICE, "Found soap server %s\n", g_str_soap_addr);
				} else if (!strcasecmp(v->name, "rtpip")) {
#if ASTERISK_VERSION_NUM >= 10600
					if (ast_parse_arg(v->value, PARSE_INADDR, &rtpaddr)) {
						ast_log(LOG_WARNING, "Invalid Sangoma RTP IP: %s\n", v->value);
						rtpaddr.sin_addr.s_addr = 0;
						continue;
					}
#else
					struct ast_hostent ahp;
					struct hostent *hp;
					if (!(hp = ast_gethostbyname(v->value, &ahp))) {
						ast_log(LOG_WARNING, "Invalid Sangoma RTP IP, could not resolve: %s\n", v->value);
						continue;
					} else {
						memcpy(&rtpaddr.sin_addr, hp->h_addr, sizeof(rtpaddr.sin_addr));
					}
#endif
					g_rtpip = ntohl(rtpaddr.sin_addr.s_addr);
					ast_log(LOG_DEBUG, "Using RTP IP %s (0x%08X)\n", v->value, g_rtpip);
				} else {
					ast_log(LOG_WARNING, "Ignored unknown Sangoma codec setting %s in [general]\n", v->name);
				}
			}
			if (!g_rtpip) {
#if ASTERISK_VERSION_NUM >= 10800
				nulladdr.sin_family = AF_INET;
				nulladdr.sin_addr.s_addr = INADDR_ANY; 
				ast_sockaddr_from_sin(&nulladdr_s, &nulladdr);
				ast_find_ourip(&rtpaddr_s, &nulladdr_s, AST_AF_INET);
#else
				ast_find_ourip(&rtpaddr.sin_addr, nulladdr);
#endif
				if (!strcasecmp(ast_inet_ntoa(rtpaddr.sin_addr), "127.0.0.1")) {
					ast_log(LOG_ERROR, "Asterisk resolved our local IP as 127.0.0.1 but that is not good enough for our RTP transcoding, you will need to configure the rtpip parameter in %s\n", SANGOMA_TRANSCODE_CONFIG);
					ret = -1;
					goto done;
				}
				g_rtpip = ntohl(rtpaddr.sin_addr.s_addr);
				ast_log(LOG_NOTICE, "No rtpip address found in configuration, using %s\n", ast_inet_ntoa(rtpaddr.sin_addr));
			}
		} else {
			ast_log(LOG_WARNING, "Ignoring unknown configuration section '%s'\n", category);
		}
	}

	if (!g_codec_register_list_count) {
		ast_log(LOG_NOTICE, "No 'register' codecs where specified in configuration file, defaulting to g729 only!\n");
		ast_copy_string(g_codec_register_list[0], "g729", sizeof(g_codec_register_list[0]));
		g_codec_register_list_count = 1;
	}

done:
	ast_config_destroy(cfg);

	g_init_cfg.host_nic_vocallo_sz = 0;

	return ret;
}

/*! \brief standard module stuff */

#if ASTERISK_VERSION_NUM >= 10400
static 
#endif
int reload(void)
{
	if (sangoma_parse_config(1)) {
		return AST_MODULE_LOAD_DECLINE;
	}
	return AST_MODULE_LOAD_SUCCESS;
}

static void unregister_translators(void)
{
	struct sangoma_translator *cur;
	ast_log(LOG_DEBUG, "Unregistering all translators\n");
	AST_LIST_LOCK(&g_sangoma_translators);
	AST_LIST_TRAVERSE_SAFE_BEGIN(&g_sangoma_translators, cur, entry) {
		QUIET_LINKED_LIST;
		ast_unregister_translator(&cur->t);
		ast_free(cur);
	}
	AST_LIST_TRAVERSE_SAFE_END;
	AST_LIST_UNLOCK(&g_sangoma_translators);
}

#if ASTERISK_VERSION_NUM >= 10400
static 
#endif
int unload_module(void)
{
	ast_cli_unregister_multiple(g_sangoma_cli, ARRAY_LEN(g_sangoma_cli));
	unregister_translators();
	sngtc_deactivate_modules();
	if (g_sched) {
		sched_context_destroy(g_sched);
		g_sched = NULL;
	}
	return 0;
}

static int sangoma_logger(int level, char *fmt, ...)
{
	char *data;
	int ret = 0;
	va_list ap;

	va_start(ap, fmt);
	ret = ast_vasprintf(&data, fmt, ap);
	if (ret == -1) {
		return ret;
	}
	va_end(ap);

	switch (level) {
	case SNGTC_LOGLEVEL_DEBUG:
		ast_log(LOG_DEBUG, "%s\n", data);  
		break;
	case SNGTC_LOGLEVEL_WARN:
		ast_log(LOG_WARNING, "%s\n", data);  
		break;
	case SNGTC_LOGLEVEL_INFO:
		ast_log(LOG_NOTICE,"%s\n", data);
		break;
	case SNGTC_LOGLEVEL_STATS:
		break;
	case SNGTC_LOGLEVEL_ERROR:
		ast_log(LOG_ERROR,"%s\n", data);
		break;
	case SNGTC_LOGLEVEL_CRIT:
		ast_log(LOG_ERROR,"%s\n", data);
		break;
	default:
		ast_log(LOG_ERROR,"Error: invalid loglevel %i\n",level);
		return -1;
	}

	ast_free(data);
	return 0;
}

static int register_translators(void)
{
	int ret = 0;
	struct sangoma_translator *cur;
	ast_log(LOG_DEBUG, "Registering all translators\n");
	AST_LIST_LOCK(&g_sangoma_translators);
	AST_LIST_TRAVERSE_SAFE_BEGIN(&g_sangoma_translators, cur, entry) {
		QUIET_LINKED_LIST;
		if (ast_register_translator(&cur->t)) {
			ast_log(LOG_ERROR, "Failed to register translator %s\n", cur->t.name);
			ret = -1;
			break;
		}
	}
	AST_LIST_TRAVERSE_SAFE_END;
	AST_LIST_UNLOCK(&g_sangoma_translators);
	return ret;
}

static void attempt_unload_conflicting_translators(sangoma_format_id srcfmt, sangoma_format_id dstfmt)
{
	int steps = 0;
	struct ast_trans_pvt *path;
	char cname[50];
#if ASTERISK_VERSION_NUM < 100000
	sangoma_format_id src_format = srcfmt;
	sangoma_format_id dst_format = dstfmt;
#else
	struct ast_format _src_format;
	struct ast_format _dst_format;
	struct ast_format *src_format = NULL;
	struct ast_format *dst_format = NULL;
	ast_format_from_old_bitfield(&_src_format, (1 << srcfmt));
	ast_format_from_old_bitfield(&_dst_format, (1 << dstfmt));
	src_format = &_src_format;
	dst_format = &_dst_format;
#endif
retryunload:
	steps = ast_translate_path_steps(dst_format, src_format);
	if (steps == 1) {
		path = ast_translator_build_path(dst_format, src_format);
		if (!path) {
			ast_log(LOG_ERROR, "Wow, steps from %s to %s but no translation path\n",
				ast_getformatname(src_format), ast_getformatname(dst_format));
			return;
		}
		/* this is evil and will cause the owner modules for this translator to 
		 * fail to unregister it but is better than crashing */
		ast_copy_string(cname, path->t->name, sizeof(cname));
		if (ast_unregister_translator(path->t)) {
			ast_log(LOG_ERROR, "Failed to force unregistration of codec %s, crash may occur!", cname);
		} else {
			ast_log(LOG_NOTICE, "Forced unregistration of codec %s\n", cname);
		}
		ast_translator_free_path(path);
		goto retryunload;
	}
}

static int in_list(char codec_list[MAX_CODECS_LIST][MAX_CODEC_STR], size_t size, const char *value, int explicit)
{
	int i;
	for (i = 0; i < size; i++) {
		if (!strcasecmp(value, codec_list[i])) {
			return 1;
		}
		/* if any list element has "all" and explicit search has not been requested 
		   then accept the value */
		if (!explicit && !strcasecmp(codec_list[i], "all")) {
			return 1;
		}
	}
	return 0;
}

static int print_list(char codec_list[MAX_CODECS_LIST][MAX_CODEC_STR], size_t size, const char *name)
{
	int i;
	ast_log(LOG_DEBUG, "%s list has %zd elements: \n", name, size);
	for (i = 0; i < size; i++) {
		ast_log(LOG_DEBUG, "%s\n", codec_list[i]);
	}
	return 0;
}

static int add_translator(sangoma_format_id srcfmt, sangoma_format_id dstfmt)
{
	struct sangoma_translator *vt;
	char tcname[255];
#if ASTERISK_VERSION_NUM < 100000
	sangoma_format_id src_format = srcfmt;
	sangoma_format_id dst_format = dstfmt;
	unsigned src_format_id = srcfmt;
	unsigned dst_format_id = dstfmt;
#else
	struct ast_format _src_format;
	struct ast_format _dst_format;
	struct ast_format *src_format = NULL;
	struct ast_format *dst_format = NULL;
	ast_format_from_old_bitfield(&_src_format, srcfmt);
	ast_format_from_old_bitfield(&_dst_format, dstfmt);
	src_format = &_src_format;
	dst_format = &_dst_format;
	unsigned src_format_id = src_format->id;
	unsigned dst_format_id = dst_format->id;
#endif

	snprintf(tcname, sizeof(tcname), "%sto%s", ast_getformatname(src_format), ast_getformatname(dst_format));
	ast_log(LOG_DEBUG, "Attempting to register translator %s (bitfield: %llu -> %llu) (id: %u -> %u)\n",
			tcname, (unsigned long long)srcfmt, (unsigned long long)dstfmt, src_format_id, dst_format_id);
	if (!in_list(g_codec_register_list, g_codec_register_list_count, ast_getformatname(src_format), 0) &&
	    !in_list(g_codec_register_list, g_codec_register_list_count, ast_getformatname(dst_format), 0) &&
	    !in_list(g_codec_register_list, g_codec_register_list_count, tcname, 0)
	   ) {
		/* neither srcfmt, dstfmt or the full translator names were found in the register list */
		ast_log(LOG_DEBUG, "Not registering translator %s because was not found in the register list\n", tcname);
		return 0;
	}
	if (in_list(g_codec_noregister_list, g_codec_noregister_list_count, ast_getformatname(src_format), 1) ||
	    in_list(g_codec_noregister_list, g_codec_noregister_list_count, ast_getformatname(dst_format), 1) ) {
		/* if the src or dest formats are in the noregister, the only way to proceeed is that 
		 * the explicit codec is in the register list */
		if (!in_list(g_codec_register_list, g_codec_register_list_count, tcname, 1)) {
			ast_log(LOG_DEBUG, "Not registering translator %s because it's src/dst was found "
					"in the noregister list and was not found explicitly in the register list\n", tcname);
			return 0;
		}
	}
		    
	if (in_list(g_codec_noregister_list, g_codec_noregister_list_count, tcname, 1)) {
		/* neither srcfmt, dstfmt or the full translator names were found in the register list */
		ast_log(LOG_DEBUG, "Not registering translator %s because it was found explicitly in the noregister list\n", tcname);
		return 0;
	}

	if (!(vt = ast_calloc(1, sizeof(*vt)))) {
		return -1;
	}
#if ASTERISK_VERSION_NUM < 100000
	vt->t.srcfmt = srcfmt;
	vt->t.dstfmt = dstfmt;
#else
	ast_format_copy(&vt->t.src_format, src_format);
	ast_format_copy(&vt->t.dst_format, dst_format);
#endif
	vt->t.framein = sangoma_framein;
	vt->t.frameout = sangoma_frameout;
	vt->t.sample = sangoma_fakesrc_sample;
#if ASTERISK_VERSION_NUM >= 10400
	vt->t.desc_size = sizeof(struct sangoma_transcoding_session);
	vt->t.newpvt = sangoma_new;
	vt->t.destroy = sangoma_destroy;
	vt->t.buf_size = SANGOMA_TRANSLATOR_BUFFER_SIZE;
#else
#error "FIXME: Asterisk 1.2 not supported yet"
	vt->t.newpvt = i == 0 ? sangoma_new_g729toulaw : sangoma_new_ulawtog729;
	vt->t.destroy = sangoma_destroy_pvt_old;
#endif
	snprintf((char *)vt->t.name, sizeof(vt->t.name), "sng%s", tcname);

	if (g_attempt_unload_conflicting_translators) {
		attempt_unload_conflicting_translators(srcfmt, dstfmt);
	}

	/* it is important to NOT register the translator just yet, just add it to our linked list 
	 * reason being that attempt_unload_conflicting_translators() has better chances to succed
	 * if we dont register our translations in the mix */
	AST_LIST_LOCK(&g_sangoma_translators);
	AST_LIST_INSERT_HEAD(&g_sangoma_translators, vt, entry);
	AST_LIST_UNLOCK(&g_sangoma_translators);

	return 0;
}

static int entry_callback(const char *module, const char *desc, int usecnt, const char *like)
{
	if (strcasestr(module, "res_timing_dahdi")) {
		g_dahdi_timing = 1;
		return 1;
	}

	if (strcasestr(module, like)) {
		return 1;
	}

	return 0;
}

static int check_dahdi_timing(void)
{
#if ASTERISK_VERSION_NUM > 10602
	int timing_sources = 0;
	if (g_allow_nondahdi_timing) {
		return 0;
	}
	timing_sources = ast_update_module_list(entry_callback, "res_timing_");
	if (timing_sources != 1 || !g_dahdi_timing) {
		return -1;
	}
#endif
	if (0) {
		entry_callback(NULL, NULL, 0, NULL);
	}
	return 0;
}

#if ASTERISK_VERSION_NUM >= 10400
static 
#endif
int load_module(void)
{
	int i, j;
	int detected = 0, activated = 0;

	if (sangoma_parse_config(0)) {
		return AST_MODULE_LOAD_FAILURE;
	}

	if (check_dahdi_timing()) {
		ast_log(LOG_ERROR, "Refusing to load this module with non-dahdi timing, please disable all res_timing_*.so modules and leave res_timing_dahdi.so as the only timing source in modules.conf\n");
		ast_log(LOG_ERROR, "You can load this module with other timing sources by setting allownondahditiming=yes in %s general configuration section\n", SANGOMA_TRANSCODE_CONFIG);
		return AST_MODULE_LOAD_FAILURE;
	}

	g_init_cfg.log = sangoma_logger;
	g_init_cfg.create_rtp = sangoma_create_rtp;
	g_init_cfg.create_rtp_port = sangoma_create_rtp_port;
	g_init_cfg.destroy_rtp = sangoma_destroy_rtp;
	g_init_cfg.release_rtp_port = sangoma_release_rtp_port;

	if (!(g_sched = sched_context_create())) {
		ast_log(LOG_ERROR, "Unable to create scheduler context\n");
		return AST_MODULE_LOAD_FAILURE;
	}

	sngtc_deactivate_modules();

	if (sngtc_detect_init_modules(&g_init_cfg, &detected)) {
		ast_log(LOG_ERROR, "Failed to detect init sangoma transcoding library\n");
		goto failure;
	}
    
	if (sngtc_activate_modules(&g_init_cfg, &activated)) {
		ast_log(LOG_ERROR, "Failed to activate sangoma transcoding library\n");
		goto failure;
	}

	if (strlen(g_str_soap_addr)) {
		ast_log(LOG_NOTICE, "Using configured SOAP server %s\n", g_str_soap_addr);
		sngtc_set_soap_server_url(g_str_soap_addr);
	}

	/* add all supported translators */
	ast_log(LOG_DEBUG, "Registering translators\n");
	print_list(g_codec_register_list, g_codec_register_list_count, "register");
	print_list(g_codec_noregister_list, g_codec_noregister_list_count, "noregister");
	for (i = 0; i < sng_array_len(g_codec_map); i++) {

		for (j = i; j < sng_array_len(g_codec_map); j++) {

			if (g_codec_map[i].codec_id == g_codec_map[j].codec_id) {
				continue;
			}

			if (add_translator(g_codec_map[i].astformat, g_codec_map[j].astformat)) {
				goto failure;
			}

			if (add_translator(g_codec_map[j].astformat, g_codec_map[i].astformat)) {
				goto failure;
			}
		}
	}

	/* register all translators with asterisk */
	if (register_translators()) {
		goto failure;
	}

	/* register CLI helpers */
	ast_cli_register_multiple(g_sangoma_cli, ARRAY_LEN(g_sangoma_cli));

	return AST_MODULE_LOAD_SUCCESS;

failure:
	unregister_translators();
	sched_context_destroy(g_sched);
	g_sched = NULL;
	return AST_MODULE_LOAD_FAILURE;
}

#if ASTERISK_VERSION_NUM >= 10400
AST_MODULE_INFO(ASTERISK_GPL_KEY, AST_MODFLAG_DEFAULT, g_sangoma_module_description,
		.load = load_module,
		.unload = unload_module,
		.reload = reload,
);
#else
char *description(void)
{
	return g_sangoma_module_description;
}

int usecount(void)
{
	int res;
	STANDARD_USECOUNT(res);
	return res;
}

char *key()
{
	return ASTERISK_GPL_KEY;
}
#endif

/* For Emacs:
 * Local Variables:
 * mode:c
 * indent-tabs-mode:t
 * tab-width:4
 * c-basic-offset:4
 * End:
 * For VIM:
 * vim:set softtabstop=4 shiftwidth=4 tabstop=4
 */
